/*
* ColorMatrix
* Visit http://createjs.com/ for documentation, updates and examples.
*
* Copyright (c) 2010 gskinner.com, inc.
* 
* Permission is hereby granted, free of charge, to any person
* obtaining a copy of this software and associated documentation
* files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use,
* copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following
* conditions:
* 
* The above copyright notice and this permission notice shall be
* included in all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
* HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
* WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*/
 
// namespace:
this.createjs = this.createjs||{};
 
(function() {
	
	/**
	 * Provides helper functions for assembling a matrix for use with the {{#crossLink "ColorMatrixFilter"}}{{/crossLink}},
	 * or can be used directly as the matrix for a ColorMatrixFilter. Most methods return the instance to facilitate
	 * chained calls.
	 *
	 * <h4>Example</h4>
	 *      myColorMatrix.adjustHue(20).adjustBrightness(50);
	 *
	 * See {{#crossLink "Filter"}}{{/crossLink}} for an example of how to apply filters, or {{#crossLink "ColorMatrixFilter"}}{{/crossLink}}
	 * for an example of how to use ColorMatrix to change a DisplayObject's color.
	 * @class ColorMatrix
	 * @constructor
	 * @extends Array
	 * @param {Number} brightness
	 * @param {Number} contrast
	 * @param {Number} saturation
	 * @param {Number} hue
	 **/
	ColorMatrix = function(brightness, contrast, saturation, hue) {
	  this.initialize(brightness, contrast, saturation, hue);
	};
	var p = ColorMatrix.prototype = [];
	
	/**
	 * Array of delta values for contrast calculations.
	 * @property DELTA_INDEX
	 * @type Array
	 * @protected
	 * @static
	 **/
	ColorMatrix.DELTA_INDEX = [
		0,    0.01, 0.02, 0.04, 0.05, 0.06, 0.07, 0.08, 0.1,  0.11,
		0.12, 0.14, 0.15, 0.16, 0.17, 0.18, 0.20, 0.21, 0.22, 0.24,
		0.25, 0.27, 0.28, 0.30, 0.32, 0.34, 0.36, 0.38, 0.40, 0.42,
		0.44, 0.46, 0.48, 0.5,  0.53, 0.56, 0.59, 0.62, 0.65, 0.68, 
		0.71, 0.74, 0.77, 0.80, 0.83, 0.86, 0.89, 0.92, 0.95, 0.98,
		1.0,  1.06, 1.12, 1.18, 1.24, 1.30, 1.36, 1.42, 1.48, 1.54,
		1.60, 1.66, 1.72, 1.78, 1.84, 1.90, 1.96, 2.0,  2.12, 2.25, 
		2.37, 2.50, 2.62, 2.75, 2.87, 3.0,  3.2,  3.4,  3.6,  3.8,
		4.0,  4.3,  4.7,  4.9,  5.0,  5.5,  6.0,  6.5,  6.8,  7.0,
		7.3,  7.5,  7.8,  8.0,  8.4,  8.7,  9.0,  9.4,  9.6,  9.8, 
		10.0
	];
	
	/**
	 * Identity matrix values.
	 * @property IDENTITY_MATRIX
	 * @type Array
	 * @protected
	 * @static
	 **/
	ColorMatrix.IDENTITY_MATRIX = [
		1,0,0,0,0,
		0,1,0,0,0,
		0,0,1,0,0,
		0,0,0,1,0,
		0,0,0,0,1
	];
	
	/**
	 * The constant length of a color matrix.
	 * @property LENGTH
	 * @type Number
	 * @protected
	 * @static
	 **/
	ColorMatrix.LENGTH = ColorMatrix.IDENTITY_MATRIX.length;
	
	
	/**
	 * Initialization method.
	 * @method initialize
	 * @protected
	 */
	p.initialize = function(brightness,contrast,saturation,hue) {
		this.reset();
		this.adjustColor(brightness,contrast,saturation,hue);
		return this;
	};
	
	/**
	 * Resets the matrix to identity values.
	 * @method reset
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 */
	p.reset = function() {
		return this.copyMatrix(ColorMatrix.IDENTITY_MATRIX);
	};
	
	/**
	 * Shortcut method to adjust brightness, contrast, saturation and hue.
	 * Equivalent to calling adjustHue(hue), adjustContrast(contrast),
	 * adjustBrightness(brightness), adjustSaturation(saturation), in that order.
	 * @method adjustColor
	 * @param {Number} brightness
	 * @param {Number} contrast
	 * @param {Number} saturation
	 * @param {Number} hue
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.adjustColor = function(brightness,contrast,saturation,hue) {
		this.adjustHue(hue);
		this.adjustContrast(contrast);
		this.adjustBrightness(brightness);
		return this.adjustSaturation(saturation);
	};
	
	/**
	 * Adjusts the brightness of pixel color by adding the specified value to the red, green and blue channels.
	 * Positive values will make the image brighter, negative values will make it darker.
	 * @method adjustBrightness
	 * @param {Number} value A value between -255 & 255 that will be added to the RGB channels.
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.adjustBrightness = function(value) {
		if (value == 0 || isNaN(value)) { return this; }
		value = this._cleanValue(value,255);
		this._multiplyMatrix([
			1,0,0,0,value,
			0,1,0,0,value,
			0,0,1,0,value,
			0,0,0,1,0,
			0,0,0,0,1
		]);
		return this;
	},
	
	/**
	 * Adjusts the contrast of pixel color.
	 * Positive values will increase contrast, negative values will decrease contrast.
	 * @method adjustContrast
	 * @param {Number} value A value between -100 & 100.
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.adjustContrast = function(value) {
		if (value == 0 || isNaN(value)) { return this; }
		value = this._cleanValue(value,100);
		var x;
		if (value<0) {
			x = 127+value/100*127;
		} else {
			x = value%1;
			if (x == 0) {
				x = ColorMatrix.DELTA_INDEX[value];
			} else {
				x = ColorMatrix.DELTA_INDEX[(value<<0)]*(1-x)+ColorMatrix.DELTA_INDEX[(value<<0)+1]*x; // use linear interpolation for more granularity.
			}
			x = x*127+127;
		}
		this._multiplyMatrix([
			x/127,0,0,0,0.5*(127-x)/255,
			0,x/127,0,0,0.5*(127-x)/255,
			0,0,x/127,0,0.5*(127-x)/255,
			0,0,0,1,0,
			0,0,0,0,1
		]);
		return this;
	};
	
	/**
	 * Adjusts the color saturation of the pixel.
	 * Positive values will increase saturation, negative values will decrease saturation (trend towards greyscale).
	 * @method adjustSaturation
	 * @param {Number} value A value between -100 & 100.
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.adjustSaturation = function(value) {
		if (value == 0 || isNaN(value)) { return this; }
		value = this._cleanValue(value,100);
		var x = 1+((value > 0) ? 3*value/100 : value/100);
		var lumR = 0.3086;
		var lumG = 0.6094;
		var lumB = 0.0820;
		this._multiplyMatrix([
			lumR*(1-x)+x,lumG*(1-x),lumB*(1-x),0,0,
			lumR*(1-x),lumG*(1-x)+x,lumB*(1-x),0,0,
			lumR*(1-x),lumG*(1-x),lumB*(1-x)+x,0,0,
			0,0,0,1,0,
			0,0,0,0,1
		]);
		return this;
	};
	
	
	/**
	 * Adjusts the hue of the pixel color.
	 * @method adjustHue
	 * @param {Number} value A value between -180 & 180.
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.adjustHue = function(value) {
		if (value == 0 || isNaN(value)) { return this; }
		value = this._cleanValue(value,180)/180*Math.PI;
		var cosVal = Math.cos(value);
		var sinVal = Math.sin(value);
		var lumR = 0.213;
		var lumG = 0.715;
		var lumB = 0.072;
		this._multiplyMatrix([
			lumR+cosVal*(1-lumR)+sinVal*(-lumR),lumG+cosVal*(-lumG)+sinVal*(-lumG),lumB+cosVal*(-lumB)+sinVal*(1-lumB),0,0,
			lumR+cosVal*(-lumR)+sinVal*(0.143),lumG+cosVal*(1-lumG)+sinVal*(0.140),lumB+cosVal*(-lumB)+sinVal*(-0.283),0,0,
			lumR+cosVal*(-lumR)+sinVal*(-(1-lumR)),lumG+cosVal*(-lumG)+sinVal*(lumG),lumB+cosVal*(1-lumB)+sinVal*(lumB),0,0,
			0,0,0,1,0,
			0,0,0,0,1
		]);
		return this;
	};
	
	/**
	 * Concatenates (multiplies) the specified matrix with this one.
	 * @method concat
	 * @param {Array} matrix An array or ColorMatrix instance.
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.concat = function(matrix) {
		matrix = this._fixMatrix(matrix);
		if (matrix.length != ColorMatrix.LENGTH) { return this; }
		this._multiplyMatrix(matrix);
		return this;
	};
	
	/**
	 * Returns a clone of this ColorMatrix.
	 * @method clone
	 * @return {ColorMatrix} A clone of this ColorMatrix.
	 **/
	p.clone = function() {
		return new ColorMatrix(this);
	};
	
	/**
	 * Return a length 25 (5x5) array instance containing this matrix's values.
	 * @method toArray
	 * @return {Array} An array holding this matrix's values.
	 **/
	p.toArray = function() {
		return this.slice(0,ColorMatrix.LENGTH);
	};
	
	/**
	 * Copy the specified matrix's values to this matrix.
	 * @method copyMatrix
	 * @param {Array} matrix An array or ColorMatrix instance.
	 * @return {ColorMatrix} The ColorMatrix instance the method is called on (useful for chaining calls.)
	 **/
	p.copyMatrix = function(matrix) {
		var l = ColorMatrix.LENGTH;
		for (var i=0;i<l;i++) {
			this[i] = matrix[i];
		}
		return this;
	};
	
// private methods:
	
	/**
	 * @method _multiplyMatrix
	 * @protected
	 **/
	p._multiplyMatrix = function(matrix) {
		var col = [];
		
		for (var i=0;i<5;i++) {
			for (var j=0;j<5;j++) {
				col[j] = this[j+i*5];
			}
			for (var j=0;j<5;j++) {
				var val=0;
				for (var k=0;k<5;k++) {
					val += matrix[j+k*5]*col[k];
				}
				this[j+i*5] = val;
			}
		}
	};
	
	/**
	 * Make sure values are within the specified range, hue has a limit of 180, brightness is 255, others are 100.
	 * @method _cleanValue
	 * @protected
	 **/
	p._cleanValue = function(value,limit) {
		return Math.min(limit,Math.max(-limit,value));
	};
	
	// 
	/**
	 * Makes sure matrixes are 5x5 (25 long).
	 * @method _fixMatrix
	 * @protected
	 **/
	p._fixMatrix = function(matrix) {
		if (matrix instanceof ColorMatrix) { matrix = matrix.slice(0); }
		if (matrix.length < ColorMatrix.LENGTH) {
			matrix = matrix.slice(0,matrix.length).concat(ColorMatrix.IDENTITY_MATRIX.slice(matrix.length,ColorMatrix.LENGTH));
		} else if (matrix.length > ColorMatrix.LENGTH) {
			matrix = matrix.slice(0,ColorMatrix.LENGTH);
		}
		return matrix;
	};
	
	createjs.ColorMatrix = ColorMatrix;
 
}());
 
    // JavaScript Document
/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2013 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by all applicable intellectual 
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

this.svgExporter = this.svgExporter || {};

(function(){
	
	function ColorTransform()
	{
		this.redMultiplier = 1.0;
		this.greenMultiplier = 1.0;
		this.blueMultiplier = 1.0;
		this.alphaMultiplier = 1.0;
		this.redOffset = 0.0;
		this.greenOffset = 0.0;
		this.blueOffset = 0.0;
		this.alphaOffset = 0.0; 						 
	}
	
	ColorTransform.prototype.Concat = function( inColorTransform ) 
	{
		this.alphaOffset = inColorTransform.alphaOffset + (inColorTransform.alphaMultiplier * this.alphaOffset);
		this.alphaMultiplier = (inColorTransform.alphaMultiplier * this.alphaMultiplier);
	
		this.redOffset = inColorTransform.redOffset + (inColorTransform.redMultiplier * this.redOffset);
		this.redMultiplier = (inColorTransform.redMultiplier * this.redMultiplier);
	
		this.greenOffset = inColorTransform.greenOffset + (inColorTransform.greenMultiplier * this.greenOffset);
		this.greenMultiplier = (inColorTransform.greenMultiplier * this.greenMultiplier);
	
		this.blueOffset = inColorTransform.blueOffset + (inColorTransform.blueMultiplier * this.blueOffset);
		this.blueMultiplier = (inColorTransform.blueMultiplier * this.blueMultiplier);	
		
		return this;		
	} 
	
	
	ColorTransform.prototype.HasColorTransform = function()
	{
		if ( 	this.alphaOffset != 0 
			||  this.alphaMultiplier != 1.0
			||	this.redOffset != 0 
			||  this.redMultiplier != 1.0
			||	this.greenOffset != 0 
			||  this.greenMultiplier != 1.0
			||	this.blueOffset != 0 
			||  this.blueMultiplier != 1.0 )
			{
				return true;	
			}

		return false;
	}
	
	ColorTransform.prototype.HasAlphaBias = function()
	{
		if ( this.alphaOffset != 0 )
		{
			return true;	
		}
		return false;
	}
	
	ColorTransform.prototype.ReturnUniqueName = function()
	{
		
		return 	( this.alphaOffset + "_" +
				this.alphaMultiplier + "_" +
				this.redOffset + "_" +
				this.redMultiplier + "_" +
				this.greenOffset + "_" +
				this.greenMultiplier + "_" +
				this.blueOffset + "_" +
				this.blueMultiplier);
	}



	ColorTransform.prototype.SetTransformValues = function(	alphaMultiplier, 
															alphaOffset, 
															redMultiplier,
															redOffset,
															greenMultiplier,
															greenOffset,
															blueMultiplier,
															blueOffset
														   )
	{
		this.alphaOffset = alphaOffset;
		this.alphaMultiplier = alphaMultiplier;
	
		this.redOffset = redOffset;
		this.redMultiplier = redMultiplier;
	
		this.greenOffset = greenOffset;
		this.greenMultiplier = greenMultiplier;
	
		this.blueOffset = blueOffset;
		this.blueMultiplier = blueMultiplier;
	}

    ColorTransform.prototype.TransformColor = function ( inColor )
    {
        var red = parseInt ( parseInt ( inColor.slice ( 1, 3), 16) * this.redMultiplier + ( this.redOffset * 255 ) + 0.5 );
        var green = parseInt ( parseInt ( inColor.slice ( 3, 5), 16) * this.greenMultiplier + ( this.greenOffset * 255 ) + 0.5 );
        var blue = parseInt ( parseInt ( inColor.slice (5, 7), 16) * this.blueMultiplier + ( this.blueOffset * 255 ) + 0.5);
        var opacity = 255;

        if ( inColor.length > 7 )
        {
            opacity = parseInt ( inColor.slice ( 7 ), 16 );
        }
        opacity = opacity * this.alphaMultiplier + ( this.alphaOffset * 255 ) ;

        if ( red > 255 ) red  = 255;  if ( red < 0 ) red  = 0;
        if ( green > 255 ) green  = 255;  if ( green < 0 ) green  = 0;
        if ( blue > 255 ) blue  = 255; if ( blue < 0 ) blue  = 0;
        if ( opacity > 255 ) opacity  = 255;  if ( opacity < 0 ) opacity  = 0;

        var redStr = red.toString(16) ;  if ( red < 16 ) redStr = "0"+ redStr;
        var greenStr = green.toString(16) ;  if ( green < 16 ) greenStr = "0"+ greenStr;
        var blueStr = blue.toString(16) ;  if ( blue < 16 ) blueStr = "0"+ blueStr;
        var opacityStr =  opacity.toString(16) ;  if (  opacity < 16 )  opacityStr = "0"+  opacityStr;

        var returnColor = "#" + redStr + greenStr + blueStr;
        
        if ( opacityStr != "FF" &&  opacityStr != "ff" )
        {
            returnColor += opacityStr;
        }
        
        return returnColor;
    }

	this.svgExporter.ColorTransform = ColorTransform;
	
}());

/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2013 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by all applicable intellectual 
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

this.svgExporter = this.svgExporter || {};

( function ()
{
	var NEW_LINE = "\n";
	var errorReport = "";
	var gErrorsReported;
	
	function ErrorsReported()
	{
		this.errors = { }; //a hash of already reported errors
	}
	
	ErrorsReported.prototype.AddError =  function ( error )
	{
		if (! this.errors[error] )
		{
			this.errors[error] = 1;
		}
	}
	
	ErrorsReported.prototype.IsErrorPresent = function ( error )
	{		
		if ( this.errors[error] )		
		{
			return true;
		}
		else
		{
			return false;
		}
	}
	
	function ErrorHandler()
	{
		//type
		this.ERROR = fl.getLocalizedString("$$$/SVG/IDS_Error=ERROR:");
		this.WARNING = fl.getLocalizedString("$$$/SVG/IDS_Warning=WARNING:");
		
		//Errors
		this.INVALID_FRAME = fl.getLocalizedString("$$$/SVG/IDS_INVALID_FRAME=Invalid Frame.");
		this.INVALID_OBJECT = fl.getLocalizedString("$$$/SVG/IDS_INVALID_OBJECT=Invalid Object.");
		this.MALFORMED_FILL = fl.getLocalizedString("$$$/SVG/IDS_MALFORMED_FILL=Malformed Fill Object.");
		this.CAN_NOT_EXPORT_BITMAP = fl.getLocalizedString("$$$/SVG/IDS_CANNOT_EXPORT_BITMAP=Can not export a Bitmap.");
		this.INVALID_MASK = fl.getLocalizedString("$$$/SVG/IDS_INVALID_MASK=Invalid Mask: Only ungrouped Shapes and Text are supported as Masks.");
		this.MALFORMED_FLA = fl.getLocalizedString("$$$/SVG/IDS_MALFORMED_FLA=The FLA file is malformed.");
		this.IMAGES_PATH_UNRESOLVED = fl.getLocalizedString("$$$/SVG/IDS_IMAGES_PATH_UNRESOLVED=Images export path unresolved.");
		this.IMAGE_EXISTS = fl.getLocalizedString("$$$/SVG/IDS_IMAGE_EXISTS=Bitmap with Read Only property exists in the same location.");
		//Warnings
		this.UNSUPPORTED_SVG_FEATURE = fl.getLocalizedString("$$$/SVG/IDS_UNSUPPORTED_FEATURE=Some features used in the content are not supported by the SVG Exporter.\nFor more information, see this help article http://www.adobe.com/go/learn_svg_feature_support_en");
		
		gErrorsReported = new ErrorsReported();
		
	}
	
	ErrorHandler.prototype.Report =  function ( type, message ) 
	{
		if(gErrorsReported.IsErrorPresent(message) == false)
		{
			errorReport += type + message + NEW_LINE;
			gErrorsReported.AddError(message);
		}
	}

	ErrorHandler.prototype.PrintReport =  function (  )
	{
		trace ( errorReport );
	}
	
	svgExporter.ErrorHandler =  ErrorHandler;

} () );






/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2013 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by all applicable intellectual 
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

this.svgExporter = this.svgExporter || {};

/************************************************************************
GLOBAL METHODS
************************************************************************/
var included = {};
var DELIMITER_FPRO = ";";
var MAC = (fl.version.indexOf("MAC") != -1);

function getDefaultName() {
	var name = fl.getDocumentDOM().name;
	name = name.replace(/[\"\[\]:;=,*?<>|&#%]+/g, ''); //replace bad characters
	name = name.replace(/\..*$/, ''); // remove extension, if any
	name = name + ".svg" // append .svg
	return name;
}
function deserialize(dataStr) {
	if (dataStr == null || typeof(dataStr) != "string") { return null; }
	var arr = dataStr.split(DELIMITER_FPRO);
	var l = arr.length;
	var o = {};
	for (var i=0; i<l; i+=2) {
		o[arr[i]] = arr[i+1];
	}
	return o;
}

function include(file) 
{
	if (!included) { included = {}; }
	if (included[file]) { return; }
	included[file] = true;
	fl.runScript(JSFL_PATH + file+ JAVASCRIPT_EXTENSION);
}

function includeScripts ( scripts )
{
	var temp = "";
	for ( var i = 0;  i < scripts.length; i++ )
	{
		temp += FLfile.read ( JSFL_PATH + scripts[i] + 	JAVASCRIPT_EXTENSION );
	}
	eval ( temp );
}

function deinclude() {
	for (var n in included) 
	{
		delete(this[n]); // deletes associated class if it exists
		delete(included[n]);
	}
}

function isPathAbsolute(path) {
	var pattern = new RegExp(MAC ? "^/" : "^[A-Za-z]:");
	return pattern.test(path);
}


function resolveRelativePath(target,path) {
	if (path.substr(0,7) == "file://") { return path; }
	if (path.indexOf("://") != -1) { return null; }
	var dir = target.substring(0,target.lastIndexOf("/"));
	if (path.charAt(0) == "/") {
		return dir+path;
	}
	
	while (path.charAt(0) == ".") {
		if (path.substr(0,2) == "./") {
			path = path.substr(2);
		} else if (path.substr(0,3) == "../") {
			path = path.substr(3);
			var index = dir.lastIndexOf("/");
			if (index == -1) { return null; }
			dir = dir.substring(0, index);
		} else {
			return null;
		}
	}
	
	return dir + "/" + path;
}

function updateOutputPath(path, docPathURI) {
	if (isPathAbsolute(path)) {
		path = FLfile.platformPathToURI(path);
	} else {
		var index = docPathURI.lastIndexOf("/");
		docPath = docPathURI.substring(0,index+1);
		if( fl.getDocumentDOM().isUncompressed ) { 
			if((index = docPath.lastIndexOf("/", docPath.length-2)) != -1) { 
				docPath = docPath.substring(0, index+1);
			}
		}

		path = resolveRelativePath(docPath, path);
	}
	
	return path;
}

function trace ( msg )
{
  	if ( RUN_MODE == "debug" )
	{
 		fl.trace ( msg );
  	}
}


/************************************************************************
GLOBAL Vars
************************************************************************/


/* When running stand alone modify below */
var JSFL_PATH = "file:///C|/Prfc/flashpro/src/tools/Extensibility/SVGExporter/ExtensionContent/jsfl/";
var EXPORT_URL = "c:/Prfc/SVGExporter/";
var JAVASCRIPT_EXTENSION = ".js";
var RUN_MODE = "debug";

/* Do not modify */
var FILE_PROTOCOL = "file:///";
var NETWORK_PROTOCOL = "file://";
var EXPORTED_SVG_URL = FILE_PROTOCOL+EXPORT_URL +"Export.svg";
var RESOURCES_FOLDER = "Images";
var RESOURCES_URI = FILE_PROTOCOL + EXPORT_URL + RESOURCES_FOLDER;

/************************************************************************
	Script Flow
************************************************************************/
var scripts = [ "XMLComposer", "ErrorHandler", "DOMExporter", "ColorMatrix", "ColorTransform"  ];

function export(properties, configuration)
{
	var config = deserialize(configuration);
	var props = deserialize(properties);
	
	if(props.default === "true") {
		props['filename'] = getDefaultName();
	}
	var folder = "";
	if (props['copy'] === "true")
		folder = props['imagesPath'];
		
	exportSVG("", props['filename'], props['embed'], folder, props['includeHiddenLayers']);
	
}

function exportSVG(basePath, filePath, embed, folder, hiddenLayers)
{
	if(filePath.substr(0,8) == 'file:///') {
		filePath = FLfile.uriToPlatformPath(filePath);
	}
	try{
		if(filePath == "")
		{
			EXPORTED_SVG_URL = fl.browseForFileURL ( "save", "Select SVG Path", "SVG Document (*.svg)", "svg" );
			if ( EXPORTED_SVG_URL == null )
			{
				return;	
			}
		}
		else
		{
			var doc =  fl.getDocumentDOM();
			var docPathURI = doc.pathURI;
			EXPORTED_SVG_URL = updateOutputPath(filePath, docPathURI);
		}
		
		EXPORT_URL = EXPORTED_SVG_URL.replace ( /([^\\\/]+\.\w+)$/, "");

		RESOURCES_FOLDER = folder;
		
		if(folder == "")
			RESOURCES_URI = "";
		else
			RESOURCES_URI = EXPORT_URL + RESOURCES_FOLDER; 
		PerformExport(embed, hiddenLayers);
	}
	catch(e){
	}
}

function PerformExport(embed, hiddenLayers)
{	
	var domExporter = new svgExporter.DOMExporter();
	
	var doc =  fl.getDocumentDOM();
	domExporter.Export( doc, embed /*embed images*/, hiddenLayers );
	var svgXML = domExporter.XMLString();
	
	
	FLfile.write(EXPORTED_SVG_URL, svgXML);
}



/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2013 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by all applicable intellectual 
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
this.svgExporter = this.svgExporter || {};

( function ()
{
	var XMLComposer = function()
	{
		var mXML	= [];
		var mNodes	= [];
		var mState	= XMLComposer.kState_Invalid;	
		
		return {'StartNode': _startNode, 'EndNode': _endNode, 'AddNode':  _addNode, 'AddAttribute': _addAttribute, 'WriteContent': _writeContent, 'WriteComment': _writeComment, 'ToString': _toString};
		
		function _startNode(name, bNoNewLine )
		{
			if (!name || name === undefined) return;
			
			if (mState === XMLComposer.kState_Beginning) 
				mXML.push(XMLComposer.kStr_GT);
				
			mState = XMLComposer.kState_Beginning;
			mNodes.push(name);
			
			if ( bNoNewLine )
			{
				mXML.push(XMLComposer.kStr_LT + name );  
			}
			else
			{
				mXML.push(XMLComposer.kStr_NewLine + XMLComposer.kStr_LT + name );   
			}		
		}

		function _endNode(bNoNewLine)
		{
			if (mState === XMLComposer.kState_Beginning)
			{
				if ( bNoNewLine )
				{
				   mXML.push(XMLComposer.kStr_GTClose);
				}
				else
				{
					mXML.push(XMLComposer.kStr_GTClose + XMLComposer.kStr_NewLine);
				}
				mNodes.pop();
			}
			else if (mNodes.length > 0)
			{
				if ( bNoNewLine )
				{
					mXML.push(XMLComposer.kStr_LTClose + mNodes.pop() + XMLComposer.kStr_GT);
				}
				else
				{
				  mXML.push(XMLComposer.kStr_LTClose + mNodes.pop() + XMLComposer.kStr_GT + XMLComposer.kStr_NewLine);  
				}
			}
			mState = XMLComposer.kState_Invalid;
		}
			
		function _addNode( xmlNodeString )
		{
			if (mState === XMLComposer.kState_Beginning) 
			{
				mXML.push(XMLComposer.kStr_GT);
			}
			
			mXML.push ( xmlNodeString );
			mState = XMLComposer.kState_Invalid;
		}
			
		function _addAttribute(name, val)
		{
			if (mState !== XMLComposer.kState_Beginning || !name || name === undefined) return;
			
			mXML.push(" " + name + "=\"" + val + "\"");
		}

		function _writeContent(content)
		{
			if (mState === XMLComposer.kState_Beginning) 
			{	
				mXML.push(XMLComposer.kStr_GT);
			}
			
			mXML.push(content);
			mState = XMLComposer.kState_Invalid;
		}

		function _writeComment(comment)
		{
			if (mState===XMLComposer.kState_Beginning)
			{
				mXML.push(XMLComposer.kStr_GT);
			}
			
			mXML.push(XMLComposer.kStr_NewLine + XMLComposer.kStr_BeginComment + comment + XMLComposer.kStr_EndComment);
			mState = XMLComposer.kState_Invalid;
		}

		function _toString()
		{
			return mXML.join(XMLComposer.kStr_Empty);
		}		
	}
	
	//Static Constants
	XMLComposer.kState_Invalid		= -1;
	XMLComposer.kState_Beginning	= 0;
	XMLComposer.kStr_GT				= ">";
	XMLComposer.kStr_LT				= "<";
	XMLComposer.kStr_GTClose		= "/>";
	XMLComposer.kStr_LTClose		= "</";
	
	XMLComposer.kStr_NewLine		= "\n";
	XMLComposer.kStr_Empty			= "";
	XMLComposer.kStr_BeginComment	= "<!--  ";
	XMLComposer.kStr_EndComment		= "-->";
	
	this.svgExporter.XMLComposer = XMLComposer;	
} () );


/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2013 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by all applicable intellectual 
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

this.svgExporter = this.svgExporter || {};

( function ()
{

	/* SVG Tags */
	var SVG = "svg";
	
	var XML_NAMESPACE = "xmlns";
	var XML_NAMESPACE_VAL = "http://www.w3.org/2000/svg";
	var XML_XMLNS_XLINK = "xmlns:xlink";
	var XML_XLINK = "http://www.w3.org/1999/xlink"
	var SVG_VERSION = "version";
	var SVG_VERSION_VALUE = "1.1";

	var SVG_GROUP = "g";
	var SVG_PATH = "path";
	var SVG_FILL_ATTRIB = "fill";
	
	var SVG_STROKE_ATTRIB = "stroke";
	var SVG_STROKE_WIDTH = "stroke-width";
	var SVG_STROKE_LINEJOIN = "stroke-linejoin";
	var SVG_STROKE_LINECAP = "stroke-linecap";
	var SVG_STROKE_LINECAP_BUTT = "butt";
	var SVG_STROKE_MITERTYPE = "miter";
	var SVG_STROKE_MITER_LIMIT = "stroke-miterlimit";
	var SVG_STROKE_NONE = "none";
	
	var SVG_MOVETO = "M";
	var SVG_QUADRATIC_TO = "Q";
	var SVG_LINE_TO = "L";
	var SVG_CLOSE_PATH = "Z";
	
	var SVG_NOFILL = "none";
	
	
	var SVG_FILL_OPACITY_ATTRIB = "fill-opacity";
	var SVG_STROKE_OPACITY_ATTRIB = "stroke-opacity";
	
	var SVG_PATH_DATA = "d";
	
	var SVG_TRANSFORM_ATTRIB = "transform";
	var SVG_MATRIX = "matrix";
	
	var SVG_SYMBOL = "symbol";
	var SVG_ID = "id";
	
	var SVG_USE = "use";
	var SVG_XLINK_HREF= "xlink:href";
	var SVG_DEFS = "defs";
	
	var SVG_LINEAR_GRADIENT = "linearGradient";
	var SVG_RADIAL_GRADIENT = "radialGradient";
	
	var SVG_GRADIENT_TRANSFORM = "gradientTransform";
	var SVG_GRADIENT_STOP = "stop ";
	var SVG_GRADIENT_STOP_OPACITY = "stop-opacity";
	var SVG_GRADIENT_OFFSET = "offset";
	var SVG_GRADIENT_STOP_COLOR = "stop-color";
	
	var SVG_URL = "url";
	
	var SVG_GRADIENT_UNITS = "gradientUnits";
	var SVG_USER_SPACEON_USE = "userSpaceOnUse";
	
	var SVG_GRADIENT_VECTOR_X1 = "x1";
	var SVG_GRADIENT_VECTOR_Y1 = "y1";
	var SVG_GRADIENT_VECTOR_X2 = "x2";
	var SVG_GRADIENT_VECTOR_Y2 = "y2";
	
	var SVG_SPREAD_METHOD = "spreadMethod";
	var SVG_SPREAD_PAD = "pad";
	var SVG_SPREAD_REFLECT  = "reflect"
	var SVG_SPREAD_REPEAT =  "repeat";
	
	var SVG_IMAGE = "image";
	var SVG_PATTERN = "pattern";
	var SVG_PATTERN_TRANSFORM = "patternTransform";
	var SVG_PATTERN_UNIT = "patternUnits";
	
	var SVG_WIDTH = "width";
	var SVG_HEIGHT = "height";
	var SVG_VIEWBOX = "viewBox";
	
	var SVG_PIX_UNITS = "px";
	
	var SVG_COORD_X = "x";
	var SVG_COORD_Y = "y";
	
	var SVG_OVERFLOW = "overflow"; 
	var SVG_VISIBLE = "visible";
	
	var SVG_MASK = "mask";
    var SVG_CLIP_PATH = "clipPath";
    var SVG_CLIP_PATH_ATTRIBUTE = "clip-path";

	var SVG_TEXT = "text";
	var SVG_TSPAN = "tspan";
	var SVG_FONT_FAMILY = "font-family";
	var SVG_FONT_SIZE = "font-size";
	
	var SVG_FONT_STYLE = "font-style";
	var SVG_ITALIC = "italic";
	var SVG_BOLD = "bold";
	var SVG_FONT_WEIGHT =  "font-weight"
	var SVG_LETTER_SPACING = "letter-spacing";
	
	var SVG_TEXT_WRITING_MODE = "writing-mode";
	var SVG_TEXT_WRITING_MODE_LR = "lr";
	var SVG_TEXT_WRITING_MODE_TB = "tb";
	var SVG_TEXT_GLYPH_ORIENTATION_VERTICAL = "glyph-orientation-vertical";

	var SVG_BASELINE_SHIFT = "baseline-shift";
				
	var SVG_FILTER = "filter";
	var SVG_COLOR_MATRIX_FILTER = "feColorMatrix";
	var SVG_FILTER_IN ="in";
	var SVG_SOURCE_GRAPHIC = "SourceGraphic";
	var SVG_COLOR_MATRIX_TYPE = "type";
	var SVG_MATRIX = "matrix";
	var SVG_COLOR_MATRIX_VALUES = "values";
	
	var SVG_COLOR_INTERPOLATION_FILTERS = "color-interpolation-filters";
	var SVG_SRGB = "sRGB";
	
	
	var SVG_ENABLE_BACKGROUND = "enable-background";
	var SVG_BACKGROUND_NEW = "new";
	
	var SVG_BLEND_MODE_NORMAL =  "normal";
	var SVG_BLEND_MODE_MULTIPLY = "multiply";
	var SVG_BLEND_MODE_SCREEN = "screen";
	var SVG_BLEND_MODE_DARKEN = "darken";
	var SVG_BLEND_MODE_LIGHTEN = "lighten";
	
	var SVG_SOURCE_GRAPHIC = "SourceGraphic";
	var SVG_BACKGROUND_IMAGE = "BackgroundImage";
	
	var SVG_FILTER_IN = "in";
	var SVG_FILTER_IN2 = "in2";
	
	var SVG_FILTER_RESULT = "result";
	
	var SVG_FILTER_BLEND  = "feBlend";
	var SVG_FILTER_BLEND_MODE = "mode";
	
	var SVG_RECT = "rect";
	
	var SVG_FILTER_GAUSSIAN_BLUR = "feGaussianBlur";
	var SVG_FILTER_GUASSIAN_BLUR_STDDEVIATION = "stdDeviation";
	
	var SVG_FILTER_FECOMPOSITE = "feComposite";
	var SVG_FILTER_FECOMPOSITE_OPERATOR = "operator";
	var SVG_FILTER_OPERATOR_OPERATOR_OVER = "over";
	var SVG_FILTER_OPERATOR_OPERATOR_IN = "in";
	var SVG_FILTER_OPERATOR_OPERATOR_OUT = "out";
	var SVG_FILTER_OPERATOR_OPERATOR_XOR = "xor";
	var SVG_FILTER_OPERATOR_OPERATOR_ATOP = "atop";
	
	var SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC = "arithmetic";
	var SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k1 = "k1";
	var SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k2 = "k2";
	var SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k3 = "k3";
	var SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k4 = "k4";
	
	var SVG_FILTER_FLOOD = "feFlood";
	var SVG_FLOOD_OPACITY = "flood-opacity";
	var SVG_FLOOD_COLOR = "flood-color";
	
	var SVG_FILTER_OFFSET = "feOffset";
	var SVG_FILTER_OFFSET_DX = "dx";
	var SVG_FILTER_OFFSET_DY = "dy";

    var SVG_XML_SPACE = "xml:space";
    var SVG_XML_SPACE_PRESERVE = "preserve";

    var SVG_FONT = "font";
    var SVG_FONT_FACE = "font-face";
    var SVG_FONT_FAMILY = "font-family";
    var SVG_GLYPH = "glyph";
    var SVG_GLYPH_UNICODE = "unicode";
    var SVG_FONT_UNITS_PER_EM = "units-per-em";
    var SVG_FONT_HORIZ_ADVANCE_X = "horiz-adv-x";
    var SVG_FONT_VERT_ADVANCE_Y = "vert-adv-y";
	
	var SVG_KERN_U1 = "u1";
	var SVG_KERN_U2 = "u2";
	var SVG_KERN_K = "k";
	var SVG_HKERN = "hkern";
    var SVG_VKERN = "vkern";
	
	var SVG_KERNING = "kerning";
	var SVG_KERNING_AUTO = "auto";

    var SVG_VERT_ORIGIN_X = "vert-origin-x";
    var SVG_VERT_ORIGIN_Y = "vert-origin-y";

    var SVG_ASCENT = "ascent";
    var SVG_DESCENT = "descent";
	
	var SVG_PRESERVE_ASPECT_RATIO = "preserveAspectRatio";
	var SVG_ASPECT_RATIO_NONE = "none";

	//non svg
	
	var SHAPE 				= "shape";
	var TEXT 				= "text";
	var INSTANCE 			= "instance";
	
	//fills
	var NOFILL 				= "noFill";
	var SOLIDFILL 			= "solid";
	var LINGRADIENTFILL 	= "linearGradient";
	var RADGRADIENTFILL 	= "radialGradient";
	var BITMAPFILL 			= "bitmap";
	var NOFILL				= "noFill";
	
		
	// strokes
	var NOSTROKE			= "noStroke";
	var SOLIDSTROKE			= "solid";
	var DOTTEDSTROKE		= "dotted";
	var DASHEDSTROKE		= "dashed";
	var RAGGEDSTROKE		= "ragged";
	var STIPPLEDSTROKE		= "stipple";
	var HATCHEDSTROKE		= "hatched";
	var STROKE_CAPTYPE_NONE = "none";
	
	var STROKES = "STROKES";
	var FILL = "FILL";
	
	//instance types
	var SYMBOL = "symbol";
	var BITMAP =  "bitmap";
	
	var MOVIECLIP = "movie clip";
	var GRAPHIC = "graphic";
	var BUTTON = "button";
	
	var GRADIENT_VECTOR_CONSTANT = 16384;
	
	var GRADIENT_EXTEND = "Extend";
	var GRADIENT_REPEAT = "Repeat";
	var GRADIENT_REFLECT = "Reflect";
	
	var RADIAL_GRADIENT_CX = "cx";
	var RADIAL_GRADIENT_CY = "cy";
	var RADIAL_GRADIENT_R = "r";
	var RADIAL_GRADIENT_FX = "fx";
	var RADIAL_GRADIENT_FY = "fy";
	
	var JPG = ".jpg";
	var PNG = ".png";
	
	var LAYER = "Layer";
	var LAYER_MASK = "mask";
	var LAYER_MASKED = "masked";
	var LAYER_NORMAL = "normal";
	var LAYER_GUIDE = "guide";
	var LAYER_GUIDED = "guided";
	var LAYER_FOLDER =  "folder";
	
	var MASK = "Mask";
	var PATTERN = "Pattern";
	var FILTER = "Filter";
	var GRADIENT = "Gradient";
	var FILTER_NAME = "Filter_ID";
	
	var GRAPHIC_LOOP_LOOP = "loop";
	var GRAPHIC_LOOP_PLAYONCE = "play once";
	var GRAPHIC_LOOP_SINGLE_FRAME = "single frame";

	
	var BLEND_MODE_NORMAL =  "normal";
	var BLEND_MODE_MULTIPLY = "multiply";
	var BLEND_MODE_SCREEN = "screen";
	var BLEND_MODE_DARKEN = "darken";
	var BLEND_MODE_LIGHTEN = "lighten";
	
	var IMAGE_COMPRESSION_LOSSLESS = "lossless";
	var IMAGE_COMPRESSION_PHOTO = "photo";
	
	var TWEEN_TYPE_MOTION = "motion";
	var TWEEN_TYPE_SHAPE = "shape";
	var TWEEN_TYPE_MOTION_OBJECT = "motion object";
	
	var BLUR_FILTER = "blurFilter";
	var GLOW_FILTER = "glowFilter";
	var DROP_SHADOW_FILTER = "dropShadowFilter";
	var BEVEL_FILTER = "bevelFilter";
	var ADJUSTCOLOR_FILTER = "adjustColorFilter";
	
	var FILTER_TYPE_INNER = "inner";
	var FILTER_TYPE_FULL = "full";
	var FILTER_TYPE_OUTER = "outer";
	
	var TEXT_ORIENTATION_HORIZONTAL = "horizontal";
	var TEXT_ORIENTATION_VERTICAL_LR = "vertical left to right";
	var TEXT_ORIENTATION_VERTICAL_RL = "vertical right to left";
	
	var TEXT_LINE_TYPE_SINGLE = "single line";
	var TEXT_LINE_TYPE_MULTILINE = "multiline";
	var TEXT_LINE_TYPE_MULTILINE_NO_WRAP = "multiline no wrap";
	var TEXT_LINE_TYPE_PASSWORD = "password";
	
	var TEXT_TYPE_STATIC = "static";
	var TEXT_TYPE_DYNAMIC = "dynamic";
	var TEXT_TYPE_INPUT = "input";

	var CHAR_POSITION_NORMAL = "normal";
	var CHAR_POSITION_SUBSCRIPT = "subscript";
	var CHAR_POSITION_SUPERSCRIPT = "superscript";

    var FONTNAME_BOLD = "Bold";
    var FONTNAME_ITALIC = "Italic";
    var FONTNAME_REGULAR = "Regular";

    var  DIRECTION_HORIZONTAL = "horizontal";
    var  DIRECTION_VERTICAL = "vertical";
    var  DIRECTION_ALL = "all";


	var DOCUMENT_TIMLINE_NAME = "";

    var DOCTYPE_FLASH = "Flash";
    var DOCTYPE_HTMLCANVAS = "htmlcanvas";
    var DOCTYPE_WEBGL = "webGLDoc";
	
	//global error handler used by everybody
	var errorHandler = new svgExporter.ErrorHandler();
	var unsupportedFeature = false;
	var gResourceDefinitions;
	var gGradientDefinitions;
	var gImageDefinitions;
	var gPatternDefinitions;
	var gFilterDefinitions;
	var gMaskDefinitions;
	var gShapeFillDefinitions;
	var gShapeStrokeDefinitions;
    var gFontTableDefinitions;
	
	var gInstanceInfoArray;
	
	var gMaskCounter = 0;
	var gFilterCounter = 0;
	var gGradientCounter = 0;
	var gPatternCounter = 0;
	var gEmbedImages =  false;
	var gExportedImagesMap = {};
	var gDoc;
	var gDocType;
    var gMaskMode = false;
	var gNamesUsed;
    var gBlendModeUsed = false;
	var gHiddenLayers = true;

    //Feature Matrix
    var gFiltersSupported = true;
    var gColorEffectsFullSupport = true;
    var gTextFullSupport = true;


	
	/*****************************************************************/	
	
	function NamesUsed()
	{
		this.formattedNames = { }; //a hash of formatted resource names that are already used.
	}
	
	NamesUsed.prototype.AddName =  function ( name )
	{
		if (! this.formattedNames[name] )
		{
			this.formattedNames[name] = 1;
		}
	}
	
	NamesUsed.prototype.IsNameUsed = function ( name )
	{		
		if ( this.formattedNames[name] )		
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	/*****************************************************************/
	
	function InstanceInfo ( name, colorTransform, matrix )
	{
		this.name = name;
		this.colorTransform = colorTransform;
		this.matrix = matrix;
	}
	
	/*****************************************************************/
	
	function ResourceDefinitions()
	{
		this.resourceItems = [];  //represents the library items in order, and other deinitions such as gradients
		this.originalNames = { }; //a hash of original resource names, used to check if a resource item is already added.
	}
	
	ResourceDefinitions.prototype.AddResourceItem =  function ( resourceItem )
	{
		var originalName = resourceItem.originalName;
		
		if ( this.originalNames[originalName] == null )
		{
			this.originalNames[originalName] = this.resourceItems.length;
			this.resourceItems[this.resourceItems.length] = resourceItem;
		}
	}
	
	ResourceDefinitions.prototype.IsItemPresent = function ( name )
	{		
		if ( this.originalNames[name] != null )		
		{
			return true;
		}
		else
		{
			return false;
		}
	}
	
	ResourceDefinitions.prototype.GetResourceItem = function ( name )
	{		
		var index = -1;

		if ( this.originalNames[name] != null  ) index = this.originalNames[name];

		if ( index >=  0 )
		{
			return this.resourceItems[index];
		}
		return null;
	}
	
	
	
	/*****************************************************************/
	
	function ResourceItem( )
	{
		this.xmlComposer = new svgExporter.XMLComposer();
		this.originalName = "";
		this.formattedName = "";
	}
	
	
	/*****************************************************************/
	
	/* DOMExporter, is the the class which drives export of the DOM */
		
	function DOMExporter()
	{
		
	}

	function PreprocessForExportWithoutStage(doc) {
		var timeLine = gDoc.timelines[0];
		var layerCount = timeLine.layerCount;
		if ( layerCount > 0)
		{
			var layer =  timeLine.layers[0];
			var frameCount = layer.frameCount;
			if(frameCount > 0) {
				var frame = layer.frames[0];
				if(frame.elements.length > 0){
					var element = frame.elements[0];
					if(element.elementType == SHAPE) {
							var sizeObj = new Object();
							sizeObj.width = element.width;
							sizeObj.height = element.height;
							element.x = element.width/2;
							element.y = element.height/2;
							doc.height = sizeObj.height;
							doc.width = sizeObj.width;
					}
				}
			}
		}
	}

	DOMExporter.prototype.Export = function ( doc, embedImages, hiddenLayers )
	{
		var savedMergeSelectionMode = doc.mergeForSelction;
		doc.mergeForSelection = true;
		gDoc = doc;
        gDocType = doc.type;
		gHiddenLayers = hiddenLayers;
        if ( gDocType == DOCTYPE_HTMLCANVAS )
        {
            gTextFullSupport = false;
        }
        
        if ( gDocType == DOCTYPE_WEBGL )
        {
            gFiltersSupported = false;
        }
		
		//Create the library and other definitions
		gResourceDefinitions = new ResourceDefinitions();
		gGradientDefinitions = new ResourceDefinitions();
		gImageDefinitions = new ResourceDefinitions();
		gPatternDefinitions = new ResourceDefinitions();
		gFilterDefinitions = new ResourceDefinitions();
		gShapeFillDefinitions = new ResourceDefinitions();
		gShapeStrokeDefinitions = new ResourceDefinitions();
        gFontTableDefinitions = new ResourceDefinitions();
		
		gNamesUsed = new NamesUsed();
		gMaskCounter = 0;
		gEmbedImages = embedImages;
        gBlendModeUsed = false;
		gInstanceInfoArray = [];
		gInstanceInfoArray[0] = new InstanceInfo ( DOCUMENT_TIMLINE_NAME, new svgExporter.ColorTransform(), CreateIdentityMatrix() );
		
		this.completexmlComposer = new svgExporter.XMLComposer();
		
		//Start
		this.completexmlComposer.StartNode(SVG);
		this.completexmlComposer.AddAttribute(XML_NAMESPACE, XML_NAMESPACE_VAL);
		this.completexmlComposer.AddAttribute(SVG_VERSION, SVG_VERSION_VALUE);
		this.completexmlComposer.AddAttribute ( XML_XMLNS_XLINK, XML_XLINK );
		this.completexmlComposer.AddAttribute ( SVG_PRESERVE_ASPECT_RATIO, SVG_ASPECT_RATIO_NONE);
		
		PreprocessForExportWithoutStage(doc);

		var height = doc.height;
		var width = doc.width;
			
		this.completexmlComposer.AddAttribute ( SVG_COORD_X, "0"+SVG_PIX_UNITS );
		this.completexmlComposer.AddAttribute ( SVG_COORD_Y, "0"+SVG_PIX_UNITS );
		
		this.completexmlComposer.AddAttribute ( SVG_WIDTH, width+SVG_PIX_UNITS );
		this.completexmlComposer.AddAttribute ( SVG_HEIGHT, height+SVG_PIX_UNITS );
		this.completexmlComposer.AddAttribute ( SVG_VIEWBOX, "0 0 "+ width + " "+height);
		
		this.xmlComposer = new svgExporter.XMLComposer();
		
		//Process Content
		var timeLineExporter =  new TimeLineExporter(this.xmlComposer);
		
		var timeline = gDoc.timelines[0];
		var frame = 0;
		/*if(gDoc.getTimeline())
		{
			timeline = gDoc.getTimeline();
			if(gDoc.getTimeline().currentFrame)
			{
				frame = gDoc.getTimeline().currentFrame; 
			}
		}
		*/
		timeLineExporter.Export(timeline, frame);
		//Write library items
		this.completexmlComposer.StartNode ( SVG_DEFS );
		
		for ( var numItems = 0; numItems < gGradientDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gGradientDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}
		
		for ( var numItems = 0; numItems < gImageDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gImageDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}

		for ( var numItems = 0; numItems < gPatternDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gPatternDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}
		for ( var numItems = 0; numItems < gFilterDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gFilterDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}
		for ( var numItems = 0; numItems < gShapeFillDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gShapeFillDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}
		for ( var numItems = 0; numItems < gShapeStrokeDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gShapeStrokeDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}
		for ( var numItems = 0; numItems < gResourceDefinitions.resourceItems.length; numItems++ )
		{
			this.completexmlComposer.AddNode ( gResourceDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}

        for ( var numItems = 0; numItems < gFontTableDefinitions.resourceItems.length; numItems++ )
		{
			gFontTableDefinitions.resourceItems[numItems].Export();
            this.completexmlComposer.AddNode ( gFontTableDefinitions.resourceItems[numItems].xmlComposer.ToString() );
		}

		
		this.completexmlComposer.EndNode();
		var hasIdentityMatrix;// = IsIdentityMatrix(doc.viewMatrix) ; 
		hasIdentityMatrix = true;
		if ( gBlendModeUsed || !hasIdentityMatrix)
        {
			this.completexmlComposer.StartNode ( SVG_GROUP );	
			if ( !hasIdentityMatrix )
			{
				this.completexmlComposer.AddAttribute ( SVG_TRANSFORM_ATTRIB,  GetMatrixString ( doc.viewMatrix ) );	
			}
			if ( gBlendModeUsed )
			{
            	this.completexmlComposer.AddAttribute ( SVG_ENABLE_BACKGROUND, SVG_BACKGROUND_NEW );
			}
        }

        //Write the body
		this.completexmlComposer.AddNode ( this.xmlComposer.ToString() );
		
		if ( gBlendModeUsed || !hasIdentityMatrix)
        {
            this.completexmlComposer.EndNode ();		
        }

        //End
		this.completexmlComposer.EndNode();
		
		doc.mergeForSelection = savedMergeSelectionMode;
		if(unsupportedFeature)
		{
			errorHandler.Report( errorHandler.WARNING, errorHandler.UNSUPPORTED_SVG_FEATURE);
			errorHandler.PrintReport();
		}
		
	}
	
	DOMExporter.prototype.XMLString = function ()
	{
		return (this.completexmlComposer.ToString());
	}

	this.svgExporter.DOMExporter = DOMExporter;
	
	/*****************************************************************/
	
	function FrameInfo(frame, currentFrameNumber)
	{
		this.frame = frame;
		this.currentFrameNumber = currentFrameNumber;
	}
		
	/*****************************************************************/
	
	function TimeLineExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	function IsSelected(elem) {
		var curr_doc = fl.getDocumentDOM();
		for(i=0 ; i<curr_doc.selection.length ; i++ ) {
			if(curr_doc.selection[i] == elem)
				return true;
		}
		return false;
	}

	TimeLineExporter.prototype.Export = function ( timeLine, currentFrame)
	{
		var layerCount = timeLine.layerCount;
		var layers =  timeLine.layers;
		
		var maskActive = false;
		var currentMaskId = "";
		
		if ( currentFrame < 0 ||  layerCount <= 0 )
		{
			errorHandler.Report( errorHandler.ERROR, errorHandler.INVALID_FRAME);
			return;
		}
		
		for ( var layerNum = ( layerCount - 1 ); layerNum >= 0; layerNum-- )
		{
			//We want layers with higher indices to be exported first.
			
			var layer = layers[layerNum];
			
			if ( ( layer.frameCount - 1) < currentFrame )
			{
				//curent layer is not lengthy enough
				continue;
			}
			if (gHiddenLayers == "false" &&  !(layer.visible) )
			{
				continue;	
			}
			
			if (  LAYER_MASK == layer.layerType )
			{
				maskActive = false;
			}
			else if ( LAYER_NORMAL == layer.layerType || LAYER_MASKED == layer.layerType || LAYER_GUIDED == layer.layerType )
			{
				var elements = layer.frames[currentFrame].elements;
				elements = elements.slice(0,1); // take only first elemet : HACK, for some reason there is a ghost element, so avoid the ghost

				var elementsLength = elements.length;
				
				if ( LAYER_MASKED == layer.layerType )
				{
					if ( !maskActive )
					{
						var maskExporter =  new MaskExporter ( this.xmlComposer );
						if ( layer.parentLayer.layerType != LAYER_MASK )
						{
							errorHandler.Report( errorHandler.ERROR, errorHandler.MALFORMED_FLA);	
						}
						
						
						var parentLayerFrame = layer.parentLayer.frames[currentFrame];
						var frameInfo = new FrameInfo( parentLayerFrame, currentFrame );
						maskActive = maskExporter.Export ( parentLayerFrame.elements, frameInfo, layerNum );
				
						if ( maskActive ==  true )
						{
							currentMaskId = maskExporter.GetMaskId();
						}
					}
					
					if ( maskActive )
					{
						this.xmlComposer.StartNode ( SVG_GROUP );	
						this.xmlComposer.AddAttribute ( SVG_MASK, SVG_URL +"(#"+ currentMaskId +")" );
					}
				}
				else
				{
					maskActive = false;	
				}
				
				var frameInfo = new FrameInfo( layer.frames[currentFrame], currentFrame );
				for ( var elemNum = 0; elemNum < elementsLength; elemNum++ )
				{
					var elementExporter =  new ElementExporter( this.xmlComposer );
					elementExporter.Export( elements[elemNum], frameInfo, LAYER + layerNum +"_"+elemNum );
				}
				if ( maskActive && LAYER_MASKED == layer.layerType )
				{
					this.xmlComposer.EndNode ();	
				}
			}
			else if ( LAYER_GUIDE == layer.layerType || LAYER_FOLDER == layer.layerType )
			{
				continue;
			}

		}
		
	}
	
	/*****************************************************************/
	
	function ElementExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	ElementExporter.prototype.Export = function ( element, frameInfo, elementDepth )
	{
		var tweenObj = frameInfo.frame.tweenObj;
		var currentFrame = frameInfo.currentFrameNumber;
		var filterExported =  false;
		
		var shapeGroup = false;
		
		var addAGroup =  false;
		
		var matrix = element.matrix;
		var filters = null;
        if ( gFiltersSupported )
        {
            filters = element.filters;
        }
		var tweenMatrix = null;
		
		var colorTransform = null;
		var instanceForColorMatrixExport = null;
		if (!gMaskMode && element.elementType == INSTANCE  && element.instanceType == SYMBOL)
		{
			if ( !element.visible ) return;

			colorTransform = new svgExporter.ColorTransform();
			instanceForColorMatrixExport = element;
		}

		if ( tweenObj && tweenObj.duration > 1 )
		{
			var index = currentFrame - tweenObj.startFrame;
			
			if ( tweenObj.tweenType == TWEEN_TYPE_MOTION  || tweenObj.tweenType == TWEEN_TYPE_MOTION_OBJECT )
			{
				tweenMatrix = tweenObj.getGeometricTransform(index);
				matrix = fl.Math.concatMatrix( matrix, tweenMatrix);	
                
                if ( gFiltersSupported )
                {
				    filters = tweenObj.getFilters( index ); 
                }

				if ( index && !gMaskMode && element.elementType == INSTANCE )
				{
					instanceForColorMatrixExport = tweenObj.getColorTransform(index);
				}
					
			}
			else if ( tweenObj.tweenType == TWEEN_TYPE_SHAPE )
			{
				element = tweenObj.getShape(index);
				matrix = element.matrix;
			}
		}
		
		if ( 	element.elementType == TEXT 
			|| element.elementType == INSTANCE 
			|| ( element.elementType == SHAPE && element.contours &&  element.contours.length )
			)
		{
			addAGroup = true;	
		}
			
		if ( instanceForColorMatrixExport )
		{
			colorTransform.SetTransformValues ( instanceForColorMatrixExport.colorAlphaPercent/100,
															instanceForColorMatrixExport.colorAlphaAmount/255,
															gColorEffectsFullSupport ? instanceForColorMatrixExport.colorRedPercent/100 : 1,
															gColorEffectsFullSupport ? instanceForColorMatrixExport.colorRedAmount/255 : 0,
															gColorEffectsFullSupport ? instanceForColorMatrixExport.colorGreenPercent/100 : 1,
															gColorEffectsFullSupport ? instanceForColorMatrixExport.colorGreenAmount/255 : 0,
															gColorEffectsFullSupport ? instanceForColorMatrixExport.colorBluePercent/100 : 1,
															gColorEffectsFullSupport ? instanceForColorMatrixExport.colorBlueAmount/255 : 0);
			var prevInstanceInfo = gInstanceInfoArray[gInstanceInfoArray.length-1 ];
			colorTransform.Concat ( prevInstanceInfo.colorTransform );
		}
		
		if ( addAGroup )
		{ 
			//Filters need to be applied outside the transformation. So we we apply filters here.
			if ( !gMaskMode && ( filters || (element.blendMode && BLEND_MODE_NORMAL != element.blendMode )))
			{ 
				var filterDefinition = new FiltersExporter();
				var filtersColorTransform = null;
				if ( filters )
				{
					filtersColorTransform = colorTransform;
				}

				var bound = {};
				bound.x = 0;
				bound.y = 0;
				bound.width = element.width;
				bound.height =  element.height;
				
				
				var instanceInfo = gInstanceInfoArray[gInstanceInfoArray.length - 1];
				var cumulativeMatrix = instanceInfo.matrix;

				if ( tweenMatrix )
				{
					bound = GetBoundingBox ( 0, 0, element.width, element.height, tweenMatrix );
				}
				
				bound = GetBoundingBox ( bound.x, bound.y, bound.width, bound.height, cumulativeMatrix );
				var newHeight = bound.height;
				var newWidth = bound.width;	
				
				var  filterDefinition = filterDefinition.Export ( filters, element.blendMode, filtersColorTransform, newWidth, newHeight  );
				if ( filterDefinition )
				{
					this.xmlComposer.StartNode ( SVG_GROUP );		
					this.xmlComposer.AddAttribute ( SVG_FILTER, SVG_URL + "(#" + filterDefinition.formattedName + ")" );
					
					var inverseMatrix = fl.Math.invertMatrix( cumulativeMatrix );
					matrix = fl.Math.concatMatrix( matrix, cumulativeMatrix );	 

					var transformation = GetMatrixString ( inverseMatrix ); 
					this.xmlComposer.AddAttribute ( SVG_TRANSFORM_ATTRIB, transformation );
					
					filterExported = true;
					
					if ( filtersColorTransform )
					{
						//We have already applied the color transform in the filter. Don't apply again
						filtersColorTransform.SetTransformValues ( 1,0,1,0,1,0,1,0 );
					}
				}
			}
					
			//Start a new group
			this.xmlComposer.StartNode ( SVG_GROUP );
			
			if ( element.name )
			{
				var id = FormatName ( element.name );
				gNamesUsed.AddName(id);
				this.xmlComposer.AddAttribute ( SVG_ID, id );
			}

			var transformation = GetMatrixString ( matrix ); 
			this.xmlComposer.AddAttribute ( SVG_TRANSFORM_ATTRIB, transformation );
		}
		
		switch (element.elementType) 
		{
			case SHAPE:
					{
						shapeGroup = element.isGroup;
						if (  element.contours &&  element.contours.length )
						{
							var shapeExporter = new ShapeExporter( this.xmlComposer );
							shapeExporter.Export ( element, elementDepth );
						}
						break;
					}
			case TEXT:
					{
						var textExporter =  new TextExporter ( this.xmlComposer );
						textExporter.Export ( element );
						break;
					}
			case INSTANCE:
					{
						var instanceExporter =  new InstanceExporter ( this.xmlComposer );
						instanceExporter.Export ( element, frameInfo, matrix, colorTransform );
						break;
					}
			default:
					{
						if(!unsupportedFeature)
							unsupportedFeature = true;
						break;
					}
		} //end of switch
		
		if ( addAGroup ) 
		{
			//End the group
			this.xmlComposer.EndNode();
			
			if ( filterExported )
			{
				this.xmlComposer.EndNode();
			}
		}
		
		if ( shapeGroup )
		{
			//A group is considered as a Shape. If so itereate over any other elements.
			var members = element.members;
			for(var memberNum = 0; memberNum < members.length; ++memberNum)
			{
				var elementExporter =  new ElementExporter ( this.xmlComposer);
				elementExporter.Export( members[memberNum], frameInfo, elementDepth+"_MEMBER_"+memberNum);
			}	
		}

	}
	
	/*****************************************************************/
	
	function MaskExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
		this.originalName = "";
		this.formattedName = "";
		this.prevMaskMode = false;
	}
	
	MaskExporter.prototype.BeginMask = function ( forClipPath)
	{
		gMaskCounter++;
		this.prevMaskMode = gMaskMode;
		gMaskMode = true;
		this.originalName = MASK + "_" + gMaskCounter;	
		this.formattedName = FormatName(this.originalName);
		gNamesUsed.AddName(this.formattedName);
        if ( forClipPath )
        {
		    this.xmlComposer.StartNode ( SVG_CLIP_PATH );
        }
        else
        {
            this.xmlComposer.StartNode ( SVG_MASK );
        }
		this.xmlComposer.AddAttribute ( SVG_ID, this.formattedName );
	}
	
	MaskExporter.prototype.EndMask = function ( )
	{
		gMaskMode = this.prevMaskMode;
		
		this.xmlComposer.EndNode();
	}


	MaskExporter.prototype.Export = function ( elements, frameInfo, layerNum )
	{
		var elementsLength = elements.length;
			
		if ( elementsLength > 0 )
		{
			this.BeginMask();
						
			var elementExporter = new ElementExporter( this.xmlComposer );
			elementExporter.Export ( elements[0], frameInfo, layerNum +"_MASK_"+0 );
	
			this.EndMask();
			
			return true;
			
		}
		
		return false;
	}
	
	MaskExporter.prototype.GetMaskId = function ()
	{
		return 	this.formattedName;
	}
	
	/*****************************************************************/
	
	function ShapeFillResource()
	{
		ResourceItem.call( this ); //call the parent
	}
	
	ShapeFillResource.prototype = new ResourceItem();
	ShapeFillResource.prototype.constructor = ShapeFillResource;
	
	ShapeFillResource.prototype.PrepareShapeFillResourceName = function (fillID)
	{
		var instanceInfo = gInstanceInfoArray[gInstanceInfoArray.length - 1];
		var prefix = instanceInfo.name;
		if ( prefix != "" )
		{
			prefix = prefix + "_" ;
		}	
		
		return prefix + fillID + "_" + FILL;
	}
	
	ShapeFillResource.prototype.GetShapeFillResourceIfPresent = function (shapeNumber)
	{

		var fillResourceName = this.PrepareShapeFillResourceName ( shapeNumber );
		if ( gShapeFillDefinitions.IsItemPresent (fillResourceName) )
		{
			return gShapeFillDefinitions.GetResourceItem(fillResourceName);
		}
		return null;
	}
	
	
	ShapeFillResource.prototype.ExportFill = function (shape, shapeNumber)
	{
		var fillResource = this.GetShapeFillResourceIfPresent ( shapeNumber );
		if ( fillResource ) return fillResource;
		
		this.originalName = this.PrepareShapeFillResourceName(shapeNumber);

		//Group the contours, this is required if there are holes in the shape.
		var fillVsContourMap = new FillVsContourMap( shape.contours );
		
		var fillsExported = false;
		for (var fillKey in fillVsContourMap.map ) 
		{
			fillsExported = true;
			break;
		}

		if ( fillsExported )
		{
			gShapeFillDefinitions.AddResourceItem( this );
			this.formattedName = FormatName ( this.originalName );
			gNamesUsed.AddName(this.formattedName);		

			this.xmlComposer.StartNode ( SVG_GROUP );	
			this.xmlComposer.AddAttribute ( SVG_ID, this.formattedName );
		}


		//Handle fills first
		for (var fillKey in fillVsContourMap.map ) 
		{
			var contourExporter =  new ContourExporter ( this.xmlComposer );
			contourExporter.Export( fillVsContourMap.map[fillKey] );
		}
		
		if ( fillsExported )
		{
			this.xmlComposer.EndNode ();	
			return this;
		}

		return null;
	}
	
	
	/*****************************************************************/
	
	function ShapeStrokesResource()
	{
		ResourceItem.call( this ); //call the parent
	}
	
	ShapeStrokesResource.prototype = new ResourceItem();
	ShapeStrokesResource.prototype.constructor = ShapeStrokesResource;
	
	ShapeStrokesResource.prototype.ExportStrokes = function (pathString, stroke, strokesNumber)
	{
		var instanceInfo = gInstanceInfoArray[gInstanceInfoArray.length - 1];
		var prefix = instanceInfo.name;
		if ( prefix != "" )
		{
			prefix = prefix + "_";	
		}
		
		this.originalName = prefix + strokesNumber + "_" + STROKES ;
		
		if ( gShapeStrokeDefinitions.IsItemPresent (this.originalName) )
		{
			return gShapeStrokeDefinitions.GetResourceItem(this.originalName);
		}
		
		gShapeStrokeDefinitions.AddResourceItem( this );
		this.formattedName = FormatName ( this.originalName );
		gNamesUsed.AddName(this.formattedName);		

		
		this.xmlComposer.StartNode ( SVG_PATH );					
		this.xmlComposer.AddAttribute ( SVG_ID, this.formattedName);
		//stroke attributes. What if each stroke is different in the path?
		var strokeExporter =  new StrokeExporter (  this.xmlComposer );
		strokeExporter.Export ( stroke );
		
		this.xmlComposer.AddAttribute ( SVG_FILL_ATTRIB, SVG_NOFILL );
		
		this.xmlComposer.AddAttribute ( SVG_PATH_DATA, pathString);
		this.xmlComposer.EndNode();

		return this;
	}
	
	/*****************************************************************/
	
	function ShapeExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}

	ShapeExporter.prototype.Export = function ( element, elementDepth )
	{
		
		if ( element.elementType != SHAPE )
		{
			errorHandler.Report( errorHandler.ERROR, errorHandler.INVALID_OBJECT);
			return;
		}
		
		var shape = element;
		
		var filterID = null;
		
		if ( !gMaskMode )
		{
			var instanceInfo = gInstanceInfoArray[gInstanceInfoArray.length - 1];
			var colorTransform = instanceInfo.colorTransform;
			var hasColorTransform = colorTransform.HasColorTransform();
		
			if ( hasColorTransform )
			{
				var filterDefinition = new FiltersExporter();
				var afterExport = filterDefinition.ExportColorTransform(colorTransform);
				filterID = afterExport.formattedName;
			}
		}
		
		var shapeResourceForFills = new ShapeFillResource();
		var afterExport =  shapeResourceForFills.ExportFill ( shape, elementDepth );
		if (afterExport)
		{
			this.xmlComposer.StartNode( SVG_USE );
			if ( filterID )
			{
				this.xmlComposer.AddAttribute (SVG_FILTER, SVG_URL + "(#" + filterID + ")");
			}
			this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "#"+ afterExport.formattedName );
			this.xmlComposer.EndNode();	
		}
		if ( !gMaskMode )
		{
			var strokesContoursExporter =  new StrokesContorExporter( this.xmlComposer );
			strokesContoursExporter.Export ( shape, elementDepth, filterID);
		}
	}
	
	
	/*****************************************************************/
	
	function FillVsContourMap( contours )
	{
		this.map = { };
		
		for (var contourNum = 0; contourNum < contours.length; ++contourNum) 
		{
			var contour = contours[contourNum];

			if ( contour.fill.style != NOFILL )
			{
				this.AddContourForKey ( PrepareFillName( contour.fill ), contour );
			}
		}
	}
	
	
	FillVsContourMap.prototype.AddContourForKey = function ( key, contour )
	{
	
		var mapLength = 0;
		if ( this.map[ key ] )
		{
			mapLength = this.map[ key ].length;
		}
		else
		{
			this.map[ key ] = [ ];
		}
		this.map [key][mapLength] = contour;	
	}

	/*****************************************************************/

	function StrokesContorExporter ( xmlComposer, exportForFont )
	{
		this.xmlComposer = xmlComposer;
		this.strokesCounter = 0;
		this.strokesToFillsCounter = 0;
        this.exportForFont = exportForFont;
	}
	
	StrokesContorExporter.prototype.AreStrokesEqual =  function ( stroke1, stroke2 )
	{
		if ( stroke1 == "" || stroke2 == "" )
		{
			return false;
		}
		
		
		var stroke1FillName =  null;
		if ( stroke1.style != NOSTROKE )
		{
			stroke1FillName = PrepareFillName( stroke1.shapeFill );
		}

		var stroke2FillName = null;
		
		if ( stroke2.style != NOSTROKE )
		{
			stroke2FillName = PrepareFillName( stroke2.shapeFill );
		}
		
		if (	( stroke1.style !=  stroke2.style )
				||	( stroke1FillName != stroke2FillName )
				||  ( stroke1.thickness != stroke2.thickness )
				||	( stroke1.joinType != stroke2.joinType )
				||	( stroke1.capType != stroke2.capType )
				)
		{
			return false;
		}	
		
		return true;	
		
	}
	

	StrokesContorExporter.prototype.BeginNewPath = function ( stroke )
	{
	}
	
	StrokesContorExporter.prototype.EndCurrentPath = function ( pathString, stroke, shapeNumber, filterID, invertMatrix)
	{
		if ( this.exportForFont )
        {
            this.xmlComposer.AddAttribute ( SVG_PATH_DATA,  pathString );
        }
        else
        {
            this.strokesCounter++;
		    var shapeStrokes =  new ShapeStrokesResource();
		    var afterExport = shapeStrokes.ExportStrokes ( pathString, stroke, shapeNumber+"_"+this.strokesCounter );
		
		    this.xmlComposer.StartNode( SVG_USE );
		    if ( filterID )
		    {
			    this.xmlComposer.AddAttribute ( SVG_FILTER, SVG_URL + "(#" + filterID + ")" );	
		    }
		    this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "#"+ afterExport.formattedName );
		    if ( invertMatrix )
		    {
			    this.xmlComposer.AddAttribute ( SVG_TRANSFORM_ATTRIB, GetMatrixString ( invertMatrix ) );
		    }
		    this.xmlComposer.EndNode();
        }
	}
	
	StrokesContorExporter.prototype.ShouldConvertStrokesToFills = function ( edge )
	{
		var stroke = edge.stroke;
		
		if ( (stroke.style != SOLIDSTROKE && stroke.style != NOSTROKE) || (stroke.style == SOLIDSTROKE && stroke.widthProfile))
		{
			return true;	
		}
		return false;
	}
	
	StrokesContorExporter.prototype.ExportStrokeFills = function (  shape, shapeNumber, filterID, edgeArray)
	{
		if ( edgeArray.length )
		{
			this.strokesToFillsCounter++;
			var fillResourceID = shapeNumber + "_" + STROKES + "_" + this.strokesToFillsCounter;
			
			var shapeFillResource = new ShapeFillResource();
			var tempShapeFillResource = shapeFillResource.GetShapeFillResourceIfPresent ( fillResourceID );
			
			if ( !tempShapeFillResource )
			{
				//Need to convert strokes to fills
				var shapeWithStrokesToFills = shape.shapeWithStrokesToFills( edgeArray );
				tempShapeFillResource = shapeFillResource.ExportFill( shapeWithStrokesToFills, fillResourceID);
			}
			
			if ( tempShapeFillResource )
			{
				this.xmlComposer.StartNode( SVG_USE );
				if ( filterID )
				{
					this.xmlComposer.AddAttribute (SVG_FILTER, SVG_URL + "(#" + filterID + ")");
				}
				this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "#"+ tempShapeFillResource.formattedName );
				this.xmlComposer.EndNode();	
			}
		}
	}

	StrokesContorExporter.prototype.ConvertLinesToFillsAndExport = function (  shape, shapeNumber, filterID, edgeArray)
	{		
		var currentStroke = "";
		var edgeArrayWithSameStrokes = [];
		
		for ( var edgeIndex = 0; edgeIndex < edgeArray.length; edgeIndex++)
		{
			var edge = edgeArray[edgeIndex];
			var stroke =  edge.stroke;
			var areStrokesEqual = true;
			if ( currentStroke != "")
			{
				areStrokesEqual = this.AreStrokesEqual( currentStroke, stroke );
			}
			currentStroke = stroke;
			
			if(!areStrokesEqual)
			{
				this.ExportStrokeFills(shape, shapeNumber, filterID, edgeArrayWithSameStrokes)
				edgeArrayWithSameStrokes = [];
			}
			edgeArrayWithSameStrokes[edgeArrayWithSameStrokes.length] = edge;
		}
		if ( edgeArrayWithSameStrokes.length )
		{
			this.ExportStrokeFills(shape, shapeNumber, filterID, edgeArrayWithSameStrokes);
		}
	}
	
	StrokesContorExporter.prototype.Export = function ( shape, shapeNumber, filterID )
	{
		var edges = shape.edges;
		
		/* We use this to keep track of edges already drawn */
		var lastMoveToPoint = "";
		var pathString = "";
		var lastCommand = "";
		var currentStroke = "";
		var lastEndPoint = "";
		var edgeArray = [];
		
		var strokesExported = false;
		
		var movementRequired = true;
		
		var matrix = shape.matrix;
		var invertMatrix = null;
		var hasScaleFactors = HasScaleFactors( matrix );
		if ( hasScaleFactors )
		{
			invertMatrix = fl.Math.invertMatrix ( matrix );
		}
		
		for ( var edgeNum = 0; edgeNum < edges.length; edgeNum++ )
		{
			var edge = edges[edgeNum];
			var stroke = edge.stroke;
			
			if ( stroke.style == NOSTROKE )
			{
				continue;	
			}
			
			strokesExported = true;
			
			var strokesEqual = this.AreStrokesEqual ( currentStroke, stroke );
			
			if ( pathString != "" && !strokesEqual )
			{
				this.EndCurrentPath ( pathString, currentStroke, shapeNumber, filterID, invertMatrix );
				pathString = "";
				movementRequired = true;
			}

			currentStroke = stroke;			
			
			var convertStrokesToFills = this.ShouldConvertStrokesToFills( edge ) ;
			var edgeArrayLength = edgeArray.length;

			
			if ( (convertStrokesToFills && !strokesEqual) || (!convertStrokesToFills) )
			{
				if ( edgeArrayLength )
				{
					//Need to convert strokes to fills
					this.ConvertLinesToFillsAndExport( shape, shapeNumber, filterID, edgeArray);
					edgeArray = [];
					edgeArrayLength = 0;
				}
			}
			
			if ( convertStrokesToFills )
			{
				edgeArray[edgeArrayLength] = edge;
				continue;
			}
			
			
			if (  pathString == "" )
			{
				//We require a new Path	
				this.BeginNewPath ( stroke );
			}
			
			var anchor1 = edge.getControl(0);
			var anchor2 = edge.getControl(2);
			var anchorPoint = edge.getControl(1);
			
			if ( hasScaleFactors )
			{
				anchor1 = TransformPoint ( matrix, anchor1);
				anchor2 = TransformPoint ( matrix, anchor2);
				anchorPoint = TransformPoint ( matrix, anchorPoint);
			}
			
			var connectionPresent = false;
			var startPoint = anchor1;
			var endPoint = anchor2;


			
			if ( ArePointsEqual ( lastEndPoint, startPoint ) )
			{
				connectionPresent = true;
			}
			else if( ArePointsEqual ( lastEndPoint, endPoint ) )
			{
				startPoint = anchor2;	
				endPoint = anchor1;
				connectionPresent = true;
			}

			if ( !connectionPresent )
			{
				movementRequired = true;
				
				if ( edgeNum < ( edges.length -  1)  )
				{
					var nextEdge = edges [ edgeNum + 1 ];
					if ( this.AreStrokesEqual ( currentStroke, nextEdge.stroke ) )
					{
						if ( 	ArePointsEqual ( startPoint, nextEdge.getControl(0) )  
							|| ArePointsEqual ( startPoint, nextEdge.getControl(2) ) )
						{
							startPoint = anchor2;	
							endPoint = anchor1;
						}
					}
					
				}
			}
			

			if ( movementRequired == true )
			{
				pathString += "\n"+SVG_MOVETO;
				
				pathString += " " + startPoint.x + " "+ startPoint.y;

				lastMoveToPoint = startPoint;

				lastCommand = "";
				movementRequired = false;
			}
			
			if ( edge.isLine == true )
			{
				if ( lastCommand != SVG_LINE_TO )
				{
					pathString += "\n"+SVG_LINE_TO;	
				}
				lastCommand = SVG_LINE_TO ;
			}
			else
			{
				if ( lastCommand != SVG_QUADRATIC_TO )	
				{
					pathString += "\n"+SVG_QUADRATIC_TO;		
				}
				lastCommand = SVG_QUADRATIC_TO;

				pathString += " " + anchorPoint.x + " "+ anchorPoint.y;
			}
			
			pathString += " "+endPoint.x + " "+endPoint.y;
			lastEndPoint = endPoint;

			if ( ArePointsEqual ( lastMoveToPoint, endPoint ) )
			{
				var tryToClose =  true;
				if ( edgeNum < ( edges.length -  1)  )
				{
					var nextEdge = edges [ edgeNum + 1 ];

					if ( 	ArePointsEqual ( lastMoveToPoint, nextEdge.getControl(0) )  
							|| ArePointsEqual ( lastMoveToPoint, nextEdge.getControl(2) ) )
					
					{
						if ( this.AreStrokesEqual ( currentStroke, nextEdge.stroke ) )
						{
							tryToClose = false;	
						}
					}
				}
				if ( tryToClose )
				{
					pathString +=  " "+SVG_CLOSE_PATH;	
					movementRequired = true;
				}
			}
			
		} //for each edge
		
		if ( pathString != "" )
		{
			this.EndCurrentPath ( pathString, currentStroke, shapeNumber, filterID, invertMatrix );
		}
		
		if ( edgeArray.length )
		{
			this.ConvertLinesToFillsAndExport( shape, shapeNumber, filterID, edgeArray);
		}
		return strokesExported;

	}  //function

	/*****************************************************************/
	
	function ContourExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	ContourExporter.prototype.Export = function ( contours )
	{
		var contour = contours[0];

		//Start a new path
		this.xmlComposer.StartNode ( SVG_PATH );
		
		//Fill Attributes such as fills
		var fillExporter = new FillExporter( this.xmlComposer );
		fillExporter.Export ( contour.fill );
		
		this.xmlComposer.AddAttribute ( SVG_STROKE_ATTRIB, SVG_NOFILL );

		
		/* We have separate contours for fills and strokes. For strokes we have separate paths.
		For fills we try to combine contours with similar fills */
		
		var pathString  = "";

		for ( var contourNum = 0; contourNum < contours.length; contourNum++ )
		
		{
			contour = contours[ contourNum ];
			
			var halfEdge = contour.getHalfEdge();
			var edge = halfEdge.getEdge();
			
			//create path
			
			var lastCommand = "";
			
			var startId = halfEdge.id;
			var id = -1;
			
			var point = halfEdge.getPrev().getVertex();
			
			pathString += "\n"+SVG_MOVETO;
			pathString += " " + point.x + " "+ point.y;
			
			while ( id !=  startId )
			{
				edge = halfEdge.getEdge();
				
				if ( edge.isLine == true )
				{
					if ( lastCommand != SVG_LINE_TO )
					{
						pathString += "\n"+SVG_LINE_TO;	
					}
					lastCommand = SVG_LINE_TO ;
				}
				else
				{
					if ( lastCommand != SVG_QUADRATIC_TO )	
					{
						pathString += "\n"+SVG_QUADRATIC_TO;		
					}
					lastCommand = SVG_QUADRATIC_TO;
					
					var point =  edge.getControl( 1 );
					pathString += " " + point.x + " "+ point.y;
				}
						
				var point = halfEdge.getVertex();
				pathString += " "+point.x + " "+point.y;
	
				halfEdge = halfEdge.getNext();
				id  = halfEdge.id;
			}
			
		}

		pathString +=  " "+SVG_CLOSE_PATH;
		this.xmlComposer.AddAttribute ( SVG_PATH_DATA, pathString);
		
		this.xmlComposer.EndNode();
	}

	/*****************************************************************/
	
	function FillColorExporter( )
	{
		this.fillStr = SVG_NOFILL;
		this.opacity = 1.0
	}
	
	FillColorExporter.prototype.Export = function ( fill )
	{
		this.fillStr = SVG_NOFILL;
		
		if ( gMaskMode )
		{
			this.fillStr = "#FFFFFF";
			this.opacity = 1.0;
			return;
		}
		
		switch(fill.style)
		{
			case SOLIDFILL:
				
				var fillColor = fill.color;
				
				if ( fillColor.length > 7 )  //if without alpha, we should have # and RGB each with 2 chars
				{
					this.fillStr = fillColor.slice ( 0, 7);
					 //the alpha value starts at index 7. The string is a Hex string
					this.opacity = parseInt ( fillColor.slice ( 7 ), 16 ) /  255.0; 
				}
				else
				{
					this.fillStr = fillColor;
					this.opacity = 1.0;
				}
				
				break;
		
			case NOFILL:
				break;
				
			case LINGRADIENTFILL:
			case RADGRADIENTFILL:
				{
					var fillName = PrepareFillName ( fill );
					
					if(gGradientDefinitions.IsItemPresent(fillName) == false)
					{
						var gradientDefinition =  new GradientDefinition();
						gradientDefinition.Export ( fill );
						gGradientDefinitions.AddResourceItem( gradientDefinition );
						gNamesUsed.AddName(gradientDefinition.formattedName);
						this.fillStr = SVG_URL +"(#"+ gradientDefinition.formattedName + ")";
					}
					else
					{
						var gradient = gGradientDefinitions.GetResourceItem(fillName);
						this.fillStr = SVG_URL +"(#"+ gradient.formattedName + ")";
					}
					break;
				}
				
			case BITMAPFILL:
				{
	
					var fillName = PrepareFillName ( fill );
					
					//Create and export Library Item only if it is not present
					if ( gPatternDefinitions.IsItemPresent ( fillName ) == false )
					{
						var patternDefinition =  new PatternDefintion ();

						if ( patternDefinition.Export( fill ) == false  )
						{
							return false;	
						}
						gPatternDefinitions.AddResourceItem( patternDefinition );
						gNamesUsed.AddName(patternDefinition.formattedName);
						this.fillStr = SVG_URL + "(#" + patternDefinition.formattedName + ")";
					}
					else
					{
						var pattern = gPatternDefinitions.GetResourceItem(fillName);
						this.fillStr = SVG_URL + "(#" + pattern.formattedName + ")";
					}

					break;
				}
		}	
	}
	
	FillColorExporter.prototype.FillStr = function()
	{
		return this.fillStr;
	}
	
	FillColorExporter.prototype.Opacity = function()
	{
		return this.opacity;
	}
	
	/*****************************************************************/

	function FillExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	FillExporter.prototype.Export = function ( fill )
	{
		
	    var fillColorExporter = new FillColorExporter();
		fillColorExporter.Export ( fill );
		this.xmlComposer.AddAttribute ( SVG_FILL_ATTRIB, fillColorExporter.FillStr() );

		var opacity =  fillColorExporter.Opacity();
		if ( opacity != 1.0 )
		{
			this.xmlComposer.AddAttribute ( SVG_FILL_OPACITY_ATTRIB, opacity );
		}
		
		return true;
	}
	
	
		
	/*****************************************************************/
	
	
	function StrokeExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	StrokeExporter.prototype.Export = function ( stroke )
	{
		switch(stroke.style)
		{
			case SOLIDSTROKE:
				{
					var fillColorExporter = new FillColorExporter();
				 	fillColorExporter.Export ( stroke.shapeFill );
					
					this.xmlComposer.AddAttribute ( SVG_STROKE_ATTRIB, fillColorExporter.FillStr()  );
					
					var opacity =  fillColorExporter.Opacity();
					if ( opacity != 1.0 )
					{
						this.xmlComposer.AddAttribute ( SVG_STROKE_OPACITY_ATTRIB, opacity );
					}
					
					var thickness =  stroke.thickness;
					if (  thickness < 1.0 )  thickness = 1.0;
					this.xmlComposer.AddAttribute ( SVG_STROKE_WIDTH, thickness );
					this.xmlComposer.AddAttribute ( SVG_STROKE_LINEJOIN, stroke.joinType );
					if(stroke.capType == STROKE_CAPTYPE_NONE)
						this.xmlComposer.AddAttribute ( SVG_STROKE_LINECAP, SVG_STROKE_LINECAP_BUTT );
					else
						this.xmlComposer.AddAttribute ( SVG_STROKE_LINECAP, stroke.capType );
					
					if(stroke.joinType == SVG_STROKE_MITERTYPE )
					{
						this.xmlComposer.AddAttribute ( SVG_STROKE_MITER_LIMIT, stroke.miterLimit );
					}
					break;
				}

			case NOSTROKE:
				{	
					this.xmlComposer.AddAttribute ( SVG_STROKE_ATTRIB, SVG_STROKE_NONE );
					break;
				}
		}	
	}
	
	/*****************************************************************/
		
	/*****************************************************************/
	
	function InstanceExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	InstanceExporter.prototype.ExportSymbol = function ( libItem, currentFrame, colorTransform, matrix)
	{
		var originalName = libItem.name;
		
		originalName += "_" + currentFrame;	

		if ( gMaskMode )
		{
			originalName = MASK + "_" + originalName;
		}
		
		this.PushSymbolName( originalName, colorTransform, matrix );
		
		var timeLineExporter = new TimeLineExporter ( this.xmlComposer  );
		timeLineExporter.Export ( libItem.timeline, currentFrame);
		
		this.PopSymbolName();
	}
	
	InstanceExporter.prototype.PushSymbolName = function (name, colorTransform, matrix)
	{
		var prevInstanceInfo = gInstanceInfoArray[gInstanceInfoArray.length-1 ];
		
		var instanceCumulativeMatrix = CreateIdentityMatrix();
		instanceCumulativeMatrix = fl.Math.concatMatrix( instanceCumulativeMatrix, matrix);		
		instanceCumulativeMatrix = fl.Math.concatMatrix ( instanceCumulativeMatrix, prevInstanceInfo.matrix  );

		var instanceInfo = new InstanceInfo( name, colorTransform, instanceCumulativeMatrix );

		gInstanceInfoArray[gInstanceInfoArray.length] = instanceInfo;
	}
	
	InstanceExporter.prototype.PopSymbolName = function ()
	{
		if ( gInstanceInfoArray.length )
		{
			delete 	gInstanceInfoArray[gInstanceInfoArray.length - 1];
			gInstanceInfoArray = gInstanceInfoArray.splice ( 0, gInstanceInfoArray.length-1);
		}
		
	}

	InstanceExporter.prototype.Export = function ( instance, frameInfo, matrix, colorTransform )
	{
		
		var currentFrame = frameInfo.currentFrameNumber;
		switch ( instance.instanceType )
		{
			case SYMBOL:
				{
					if(instance.is3D || instance.parameters)
					{
						if(!unsupportedFeature)
							unsupportedFeature = true;
					}
					
					var frameToExport = 0;
					if ( instance.loop )
					{
						var graphicFirstFrame = instance.firstFrame;
						var graphicNumFrames = instance.libraryItem.timeline.frameCount;

						var numFramesElapsedSinceLastKey = currentFrame - frameInfo.frame.startFrame;
						frameToExport = ( graphicFirstFrame + numFramesElapsedSinceLastKey );
						
						if ( instance.loop == GRAPHIC_LOOP_LOOP )
						{
							frameToExport = frameToExport % graphicNumFrames;
						}
						else if ( instance.loop == GRAPHIC_LOOP_PLAYONCE )
						{
							if ( frameToExport > graphicNumFrames ) frameToExport  = graphicNumFrames-1 ;
						}
						else if (  instance.loop == GRAPHIC_LOOP_SINGLE_FRAME )
						{
							frameToExport = graphicFirstFrame;
						}
					}

					this.ExportSymbol( instance.libraryItem, frameToExport, colorTransform, matrix);

				}
				break;

			case BITMAP:
				{

					if ( gMaskMode )
					{
						this.xmlComposer.StartNode ( SVG_RECT );
						this.xmlComposer.AddAttribute ( SVG_COORD_X, 0 );
						this.xmlComposer.AddAttribute ( SVG_COORD_Y, 0 );
						this.xmlComposer.AddAttribute ( SVG_WIDTH, instance.hPixels );
						this.xmlComposer.AddAttribute ( SVG_HEIGHT, instance.vPixels );
						this.xmlComposer.AddAttribute ( SVG_FILL_ATTRIB, "#FFFFFF" );
						this.xmlComposer.EndNode();		
					}
					else
					{
						var filterID = null;
					

						var prevInstanceInfo = gInstanceInfoArray[gInstanceInfoArray.length-1 ];
						var colorTransform = prevInstanceInfo.colorTransform;
				
						var filterID = null;
						if ( colorTransform.HasColorTransform() )
						{
							var filterDefinition = new FiltersExporter();
							var afterExport = filterDefinition.ExportColorTransform(colorTransform);
							filterID = afterExport.formattedName;
						}
				
						if ( filterID )
						{
							this.xmlComposer.AddAttribute (SVG_FILTER, SVG_URL + "(#" + filterID + ")");
						}
						
						var bitmapExporter = new BitmapExporter ();
						var afterExport = bitmapExporter.Export ( instance.libraryItem );
		
						if ( afterExport )
						{
							this.xmlComposer.StartNode( SVG_USE );
							this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "#"+ afterExport.formattedName );
							this.xmlComposer.EndNode();
						}	
					}
					
					break;	
				}
			default:
				{
					if(!unsupportedFeature)
						unsupportedFeature = true;
				}
		}
	}
		
	/*****************************************************************/
	
	function FiltersExporter()
	{
		ResourceItem.call( this ); //call the parent	
		this.name = "";
		this.resultCounter = 0;
	}
	
	FiltersExporter.prototype = new ResourceItem();
	FiltersExporter.prototype.constructor = FiltersExporter;

	FiltersExporter.prototype.NextResultName = function ()
	{
		this.resultCounter++;
		return SVG_FILTER_RESULT + 	this.resultCounter;
	}
	

	FiltersExporter.prototype.BeginFilter = function (x, y, width, height)
	{
		gFilterCounter++;
		this.formattedName = FormatName(FILTER + "_" + gFilterCounter);
		gNamesUsed.AddName(this.formattedName);
		gFilterDefinitions.AddResourceItem( this );


		this.xmlComposer.StartNode( SVG_FILTER );
		this.xmlComposer.AddAttribute ( SVG_ID, this.formattedName );
		
		this.xmlComposer.AddAttribute ( SVG_COORD_X, x+"%" );
		this.xmlComposer.AddAttribute ( SVG_COORD_Y, y+"%" );
		this.xmlComposer.AddAttribute ( SVG_WIDTH, width+"%" );
		this.xmlComposer.AddAttribute ( SVG_HEIGHT, height+"%" );
		
		this.xmlComposer.AddAttribute ( SVG_COLOR_INTERPOLATION_FILTERS, SVG_SRGB );
	}
	
	FiltersExporter.prototype.EndFilter = function ()
	{
		this.xmlComposer.EndNode();
	}
	
	FiltersExporter.prototype.ColorMatrixFilter = function ( xmlComposer, values, inObj )
	{
		xmlComposer.StartNode( SVG_COLOR_MATRIX_FILTER );
		
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObj );
		xmlComposer.AddAttribute ( SVG_COLOR_MATRIX_TYPE, SVG_MATRIX );
		xmlComposer.AddAttribute ( SVG_COLOR_MATRIX_VALUES, values );	
		
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );

		xmlComposer.EndNode();
		return nextResult;		
	}

	FiltersExporter.prototype.ColorMatrixObject = function ( xmlComposer, colorMatrix, inObj )
	{
		var values = "";
		for ( var i = 0;  i < 20; i++ )
		{
			values += 	colorMatrix[i] + " ";
		}
		return this.ColorMatrixFilter(xmlComposer, values, inObj );
	}
	
	FiltersExporter.prototype.OpaqueWhite = function ( xmlComposer, inObj )
	{
		var values = 	"0 0 0 0 " + 255 + " "+
						"0 0 0 0 " + 255 + " "+
						"0 0 0 0 " + 255 + " "+
						"0 0 0 255 " + 0 ;
						
		return this.ColorMatrixFilter(xmlComposer, values, inObj );
	}
	
	
	FiltersExporter.prototype.Blur = function ( xmlComposer, inObj, blurX, blurY )
	{
		xmlComposer.StartNode ( SVG_FILTER_GAUSSIAN_BLUR );
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObj );
		 /*http://en.wikipedia.org/wiki/Gaussian_blur
		 When computing a discrete approximation of the Gaussian function, pixels at a distance of
		 more than 3σ are small enough to be considered effectively zero.*/
		xmlComposer.AddAttribute ( SVG_FILTER_GUASSIAN_BLUR_STDDEVIATION, blurX/3+","+blurY/3 );
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );
		xmlComposer.EndNode();
		return nextResult;
	}

	FiltersExporter.prototype.ColorFlood = function ( xmlComposer, color, inObject )
	{
		var R = parseInt(color.slice(1,3), 16);
		var G = parseInt(color.slice(3,5), 16);
		var B = parseInt(color.slice(5,7), 16);
		var A = 1.0;
		if ( color.length > 7 )
		{
			A = parseInt(color.slice(7), 16) / 255.0  ;
		}

		xmlComposer.StartNode ( SVG_FILTER_FLOOD );
		xmlComposer.AddAttribute ( SVG_FLOOD_COLOR, color.slice(0,7) );
		xmlComposer.AddAttribute ( SVG_FLOOD_OPACITY, A );
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObject );
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );		
		xmlComposer.EndNode();
		
		return nextResult;
	}
	
	FiltersExporter.prototype.Composite = function ( xmlComposer, operator, in2Obj, inObj )
	{
		xmlComposer.StartNode ( SVG_FILTER_FECOMPOSITE );
		xmlComposer.AddAttribute ( SVG_FILTER_IN2, in2Obj );
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObj );
		xmlComposer.AddAttribute ( SVG_FILTER_FECOMPOSITE_OPERATOR, operator );
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );
		xmlComposer.EndNode();
		return nextResult;
	}
	
	FiltersExporter.prototype.CompositeArithmetic = function ( xmlComposer, in2Obj, inObj, k1, k2, k3, k4 )
	{
		xmlComposer.StartNode ( SVG_FILTER_FECOMPOSITE );

		xmlComposer.AddAttribute ( SVG_FILTER_IN2, in2Obj );
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObj );

		xmlComposer.AddAttribute ( SVG_FILTER_FECOMPOSITE_OPERATOR, SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC);
		xmlComposer.AddAttribute ( SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k1, k1);
		xmlComposer.AddAttribute ( SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k2, k2);
		xmlComposer.AddAttribute ( SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k3, k3);
		xmlComposer.AddAttribute ( SVG_FILTER_FECOMPOSITE_OPERATOR_ARITHMETIC_k4, k4);
		
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );
		xmlComposer.EndNode();
		return nextResult;
	}


	FiltersExporter.prototype.Blend = function (  xmlComposer, svgBlendMode, in2Obj, inObj )
	{
		xmlComposer.StartNode ( SVG_FILTER_BLEND );
		xmlComposer.AddAttribute ( SVG_FILTER_BLEND_MODE, svgBlendMode );
		xmlComposer.AddAttribute ( SVG_FILTER_IN2, in2Obj );
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObj );
		
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );
		
		xmlComposer.EndNode();
		
		return nextResult;
	}
	
	FiltersExporter.prototype.Offset = function ( xmlComposer, inObject, xDistance, yDistance )
	{
		xmlComposer.StartNode ( SVG_FILTER_OFFSET );
		xmlComposer.AddAttribute ( SVG_FILTER_OFFSET_DX, xDistance );
		xmlComposer.AddAttribute ( SVG_FILTER_OFFSET_DY, yDistance );
		xmlComposer.AddAttribute ( SVG_FILTER_IN, inObject );
		var nextResult = this.NextResultName();
		xmlComposer.AddAttribute ( SVG_FILTER_RESULT, nextResult );
		xmlComposer.EndNode();
		
		return nextResult;
	}	


	FiltersExporter.prototype.ExportBlendMode = function ( xmlComposer, blendMode, sourceGraphic)
	{
		var svgBlendMode = SVG_BLEND_MODE_NORMAL;
		
		//See if there are any blend modes.
		switch ( blendMode )
		{
			case BLEND_MODE_NORMAL :
			{
				//Don't do anything
				break;
			}
			case BLEND_MODE_MULTIPLY:
			{
				svgBlendMode = 	SVG_BLEND_MODE_MULTIPLY;
				break;
			}
			case BLEND_MODE_SCREEN:
			{
				svgBlendMode = 	SVG_BLEND_MODE_SCREEN;
				break;
			}
			case BLEND_MODE_DARKEN:
			{
				svgBlendMode = SVG_BLEND_MODE_DARKEN;
				break;
			}
			case BLEND_MODE_LIGHTEN:
			{
				svgBlendMode = SVG_BLEND_MODE_LIGHTEN;
				break;
			}
			
			default:
			{
				if(!unsupportedFeature)
					unsupportedFeature = true;
				return false;
			}
		}	
		
		if (  svgBlendMode == SVG_BLEND_MODE_NORMAL )
		{	
			return false;
		}
		
		var opaqueWhite = this.OpaqueWhite ( xmlComposer, sourceGraphic);
        var backGroundCut = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, SVG_BACKGROUND_IMAGE);
        this.Blend ( xmlComposer, svgBlendMode, backGroundCut, sourceGraphic );
        gBlendModeUsed = true;
		
		return true;
	}
	
	FiltersExporter.prototype.ExportColorTransform = function ( colorTransform )
	{
		this.originalName = colorTransform.ReturnUniqueName();	
		
		if ( gFilterDefinitions.IsItemPresent( this.originalName ) )
		{
			return gFilterDefinitions.GetResourceItem( this.originalName );
		}
		
		this.BeginFilter(-20, -20, 140, 140);		
		
		var values = colorTransform.redMultiplier + " 0 0 0 " + colorTransform.redOffset + " "+
			"0 " + colorTransform.greenMultiplier + " 0 0 " + colorTransform.greenOffset + " "+
			"0 0 " + colorTransform.blueMultiplier + " 0 " + colorTransform.blueOffset + " "+
			"0 0 0 " + colorTransform.alphaMultiplier + " " + colorTransform.alphaOffset ;
				
		var opaqueWhite = null;
		if ( colorTransform.HasAlphaBias() )
		{
			opaqueWhite = this.OpaqueWhite ( this.xmlComposer, SVG_SOURCE_GRAPHIC);
		}
		var result = this.ColorMatrixFilter(this.xmlComposer, values, SVG_SOURCE_GRAPHIC);
		
		if ( opaqueWhite )
		{
			this.Composite ( this.xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, result );
		}
		this.EndFilter();
		
		return this;
	}

	FiltersExporter.prototype.PrepareName  = function ( filters, blendMode, colorTransform, width, height )
	{
		var name = blendMode + "_" +width + "_"+height;
		if ( colorTransform )
		{
			name += colorTransform.ReturnUniqueName();
		}
		
		if ( filters )
		{
			for ( var filterNum = 0; filterNum < filters.length;  filterNum++ )
			{
				var filter = filters[filterNum];
				for ( var key in filter )
				{
					name += "_" + 	filter[key];
				}
			}
		}
		return name;
	}
	
	FiltersExporter.prototype.Export = function ( filters, blendMode, colorTransform, width, height )
	{

		var filterDefined = false;

		this.originalName = this.PrepareName(  filters, blendMode, colorTransform, width, height );
		if ( gFilterDefinitions.IsItemPresent( this.originalName ) )
		{
			return gFilterDefinitions.GetResourceItem( this.originalName );
		}

		var sourceGraphic = SVG_SOURCE_GRAPHIC;
		
		var xmlComposer = new svgExporter.XMLComposer();
		
		var extraWidth = 0;
		var extraHeight = 0;
		
		var yDistance = 0;
		var xDistance = 0;

		//there are some filters
		if ( filters )
		{
			for ( var filterNum = 0;  	filterNum < filters.length;  filterNum++ )
			{

				var filter = filters[filterNum];
				
				if (! filter.enabled )
					continue;
				
				if ( filter.blurX )
				{
					extraWidth += filter.blurX;
				}
				
				if ( filter.blurY )
				{
					extraHeight += filter.blurY;
				}
				
				if ( filter.distance )
				{
					var angleInRadians = filter.angle * Math.PI / 180.0;

					yDistance = Math.sin( angleInRadians ) *  filter.distance;
					xDistance = Math.cos( angleInRadians ) *  filter.distance;

					extraWidth += Math.abs(xDistance);
					extraHeight += Math.abs(yDistance);
				}


				if ( filter.name == BLUR_FILTER )
				{
					sourceGraphic = this.Blur ( xmlComposer, sourceGraphic, filter.blurX, filter.blurY );	
					filterDefined = true;
				}
				else if ( filter.name == GLOW_FILTER )
				{
					filterDefined = true;
					var colorFlood = this.ColorFlood ( xmlComposer, filter.color, sourceGraphic);

					if ( filter.inner )
					{
						var blackFlood = this.ColorFlood ( xmlComposer, "#000000", sourceGraphic);
						var blackCroppedToSource = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, sourceGraphic, blackFlood );
						var opaqueWhite = this.OpaqueWhite ( xmlComposer, blackCroppedToSource);
						
						var colorCroppedToSource = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, sourceGraphic, colorFlood );
	
						var blurResult = this.Blur ( xmlComposer, blackCroppedToSource, filter.blurX, filter.blurY );	
						//This is how we get brighter colors outside.
						var compositeResult = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OUT, blurResult, colorCroppedToSource); 
						
						if ( !filter.knockout ) 
						{
							compositeResult = this.Blend ( xmlComposer, SVG_BLEND_MODE_NORMAL, sourceGraphic, compositeResult  ); 
						}
						sourceGraphic =  this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, compositeResult );
					}
					else
					{
						if ( filter.knockout )
						{
							
							var colorCroppedToSource = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, sourceGraphic, colorFlood );
							var blurResult = this.Blur ( xmlComposer, colorCroppedToSource, filter.blurX, filter.blurY );
							sourceGraphic =  this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OUT, sourceGraphic, blurResult );
						}
						else
						{
							var blurResult = this.Blur ( xmlComposer, sourceGraphic, filter.blurX, filter.blurY );	
							var compositeResult = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, blurResult, colorFlood );
							sourceGraphic = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OVER, compositeResult, sourceGraphic );
						}
					}
				}
				else if ( filter.name == DROP_SHADOW_FILTER )
				{
					filterDefined = true;

					var colorFlood = this.ColorFlood ( xmlComposer, filter.color, sourceGraphic);	
					var compositeResult = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, sourceGraphic, colorFlood );				

					if ( filter.inner )
					{
						var opaqueWhite = this.OpaqueWhite ( xmlComposer, sourceGraphic);
						var offsetSource = this.Offset ( xmlComposer, sourceGraphic, xDistance, yDistance );
						var offsetSourceBlur = this.Blur ( xmlComposer, offsetSource,  filter.blurX, filter.blurY );
						compositeResult = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OUT, offsetSourceBlur, compositeResult );

						if ( !filter.hideObject && !filter.knockout )
						{
							var offsetWhite = this.Offset ( xmlComposer, opaqueWhite, xDistance, yDistance );
							var nShadowW = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, offsetWhite );
							var nShadowSrc = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, nShadowW, sourceGraphic );
							compositeResult = this.Blend ( xmlComposer, SVG_BLEND_MODE_NORMAL, nShadowSrc, compositeResult );
						}
						
						sourceGraphic = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, compositeResult );
					}
					else
					{
						var blurResult = this.Blur ( xmlComposer, compositeResult, filter.blurX, filter.blurY );
						var offsetResult = this.Offset ( xmlComposer, blurResult, xDistance, yDistance );

						//External Shadow
						if ( filter.knockout )
						{
							sourceGraphic = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OUT, sourceGraphic, offsetResult );
						}
						else if ( filter.hideObject )
						{
							sourceGraphic = offsetResult;	
						}
						else
						{
							sourceGraphic = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OVER, offsetResult, sourceGraphic );
						}
					}
				}
				else if ( filter.name == BEVEL_FILTER )
				{
					filterDefined = true;

					//For Shadows
					var colorFloodHighLight = this.ColorFlood ( xmlComposer, filter.highlightColor, sourceGraphic);	
					var compositeResultHighLight =this.Composite (xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN,sourceGraphic,colorFloodHighLight );
					var blurResultHighLight = this.Blur ( xmlComposer, compositeResultHighLight, filter.blurX, filter.blurY );
					
					var colorFloodShadow = this.ColorFlood ( xmlComposer, filter.shadowColor, sourceGraphic);	
					var compositeResultShadow = this.Composite (xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, sourceGraphic, colorFloodShadow );
					var blurResultShadow = this.Blur ( xmlComposer, compositeResultShadow, filter.blurX, filter.blurY );		
					
					var offsetResultHighLight = this.Offset ( xmlComposer, blurResultHighLight, xDistance, yDistance );	
					var offsetResultHighLight1 = this.Offset ( xmlComposer, blurResultHighLight, -xDistance, -yDistance );	
					
					var offsetResultShadow = this.Offset ( xmlComposer, blurResultShadow, -xDistance, -yDistance );	
					var offsetResultShadow1 = this.Offset ( xmlComposer, blurResultShadow, xDistance, yDistance );
					var opaqueWhite = this.OpaqueWhite ( xmlComposer, sourceGraphic);
	
					var bevelsInner = null;
					var bevelsOuter = null;
					
					if ( filter.type == FILTER_TYPE_INNER)
					{
						bevelsInner = this.ColorFlood(xmlComposer, "#00000000", sourceGraphic );

						var innerHighLight = this.CompositeArithmetic( xmlComposer, offsetResultHighLight1, offsetResultHighLight, 0, -1, 1, 0 );

						var innerShadow = this.CompositeArithmetic ( xmlComposer, offsetResultShadow1, offsetResultShadow, 0, -1, 1, 0 );

						bevelsInner = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_XOR, bevelsInner, innerHighLight );	
						bevelsInner = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_XOR, bevelsInner, innerShadow );				
						
						bevelsInner = this.Composite ( xmlComposer,  SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, bevelsInner );	
					}
								
					if ( filter.type == FILTER_TYPE_OUTER ||  filter.type == FILTER_TYPE_FULL )
					{
						bevelsOuter = this.ColorFlood(xmlComposer, "#00000000", sourceGraphic );
						
						var outerHighLight = this.CompositeArithmetic( xmlComposer, offsetResultHighLight1, offsetResultHighLight, 0, -1, 1, 0 );
						bevelsOuter = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_XOR, bevelsOuter, outerHighLight );
						
						var outerShadow = this.CompositeArithmetic( xmlComposer, offsetResultShadow1, offsetResultShadow, 0, -1, 1, 0 );
						bevelsOuter = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_XOR, bevelsOuter, outerShadow );
					}

					var bevels;
					if (filter.type == FILTER_TYPE_FULL)
					{
						bevels = bevelsOuter;	
					}
					else if ( bevelsInner )
					{
						bevels = bevelsInner;
					}
					else if ( bevelsOuter )
					{
						bevels = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_OUT, sourceGraphic, bevelsOuter);
					}

					if ( !filter.knockout )
					{
						bevels = this.Blend ( xmlComposer, SVG_BLEND_MODE_NORMAL, sourceGraphic, bevels);	
					}
					sourceGraphic = bevels;
				}
				else if ( ADJUSTCOLOR_FILTER == filter.name )
				{
					filterDefined = true;
					var colorMatrix = new createjs.ColorMatrix(filter.brightness/255, filter.contrast, filter.saturation, filter.hue);
					sourceGraphic = this.ColorMatrixObject ( xmlComposer, colorMatrix, sourceGraphic);
				}
				else
				{
					if(!unsupportedFeature)
						unsupportedFeature = true;
				}
			} // for each filter
		} //if filters

		if ( colorTransform && colorTransform.HasColorTransform() )
		{
			filterDefined = true;	

			var values = colorTransform.redMultiplier + " 0 0 0 " + colorTransform.redOffset + " "+
			"0 " + colorTransform.greenMultiplier + " 0 0 " + colorTransform.greenOffset + " "+
			"0 0 " + colorTransform.blueMultiplier + " 0 " + colorTransform.blueOffset + " "+
			"0 0 0 " + colorTransform.alphaMultiplier + " " + colorTransform.alphaOffset ;
				
			var opaqueWhite = null;
			if ( colorTransform.HasAlphaBias() )
			{
				opaqueWhite = this.OpaqueWhite ( xmlComposer, sourceGraphic);
			}
			
			sourceGraphic = this.ColorMatrixFilter(xmlComposer, values, sourceGraphic);
			
			if ( opaqueWhite )
			{
				sourceGraphic = this.Composite ( xmlComposer, SVG_FILTER_OPERATOR_OPERATOR_IN, opaqueWhite, result );
			}
		}

		if ( this.ExportBlendMode ( xmlComposer, blendMode, sourceGraphic ) )
		{
			filterDefined = true;	
		}
		
		if ( filterDefined )
		{
			/*extra 10 to allow for any addional strokes*/
			var xPercent = extraWidth * 100 / width + 10;  
			var yPercent = extraHeight * 100 / height + 10; 
			
			if ( xPercent < 20 ) xPercent =  20;
			if ( yPercent < 20 ) yPercent =  20;
			if ( xPercent > 100 ) xPercent =  100;
			if ( yPercent > 100 ) yPercent =  100;
			

			this.BeginFilter(-xPercent, -yPercent, 100+xPercent * 2, 100+yPercent * 2);
			this.xmlComposer.AddNode ( xmlComposer.ToString() );
			this.EndFilter();	
			
			return this;		
		}
		
		return null;
	}
	
	/*****************************************************************/
	
	function BitmapExporter()
	{
		ResourceItem.call( this ); //call the parent	
	}
	
	BitmapExporter.prototype = new ResourceItem();
	BitmapExporter.prototype.constructor = BitmapExporter;

	BitmapExporter.prototype.Export = function ( bitmapItem )
	{
		this.originalName = bitmapItem.name;

		if ( gImageDefinitions.IsItemPresent( this.originalName ) )
		{
			return gImageDefinitions.GetResourceItem( this.originalName );
		}

		this.formattedName = FormatName( this.originalName );
		gNamesUsed.AddName(this.formattedName);
		gImageDefinitions.AddResourceItem( this ) ;
		
		var imageFileName = this.formattedName;
		
		if ( !this.DoesHaveSupportedImageExtension ( imageFileName ) )
		{
			switch ( bitmapItem.originalCompressionType )
			{
				case IMAGE_COMPRESSION_LOSSLESS:
				{
					imageFileName += PNG;	
					break;
				}
				case IMAGE_COMPRESSION_PHOTO:
				default:
				{
					imageFileName += JPG;
				}
			}
		}
		this.xmlComposer.StartNode (SVG_IMAGE);

		this.xmlComposer.AddAttribute ( SVG_ID, this.formattedName);
		this.xmlComposer.AddAttribute ( SVG_COORD_X, "0");
		this.xmlComposer.AddAttribute ( SVG_COORD_Y, "0");
		this.xmlComposer.AddAttribute ( SVG_WIDTH, bitmapItem.hPixels );
		this.xmlComposer.AddAttribute ( SVG_HEIGHT, bitmapItem.vPixels );
		
		var imageURL = "";
		//Was file already exported?
		if (! gExportedImagesMap[ bitmapItem.name ]  )
		{	
			if(gEmbedImages == "true" || (RESOURCES_URI == "" && !bitmapItem.sourceFileExists))
			{
				imageURL = fl.configURI + imageFileName;
				while(FLfile.exists(imageURL))
				{
					imageURL += "_FL";
				}
			}
			else if(RESOURCES_URI != "")
			{
				if(RESOURCES_FOLDER.charAt(0) == "." )
				{
					var isValid = false;
					
					if( RESOURCES_FOLDER == "." || RESOURCES_FOLDER == "..")
						isValid = true;
					else if( RESOURCES_FOLDER.substr(0,2) == "./")
						isValid = true;
					else if( RESOURCES_FOLDER.substr(0,3) == "../")
						isValid = true;
						
					if(isValid == false)
					{
						errorHandler.Report(errorHandler.ERROR, errorHandler.IMAGES_PATH_UNRESOLVED);
						errorHandler.PrintReport();
						throw new Error('Images path is invalid.');
					}
				}
				
				if ( !FLfile.exists( RESOURCES_URI ) )
				{	
					if( !FLfile.createFolder ( RESOURCES_URI ) )
					{
						errorHandler.Report(errorHandler.ERROR, errorHandler.IMAGES_PATH_UNRESOLVED);
						errorHandler.PrintReport();
						throw new Error('Images path is invalid.');
					}
				}
				imageURL = RESOURCES_URI + "/" + imageFileName;
			}
			
			if(imageURL != "")
			{
				var index = imageURL.lastIndexOf("/");
				var folder = imageURL.substr(0, index);
				
				if(!FLfile.exists(folder))
				{
					FLfile.createFolder(folder);
				}
				
				if(FLfile.exists(imageURL))
				{
					var fileAttr = FLfile.getAttributes(imageURL);
					if (fileAttr && (fileAttr.indexOf("R") != -1))
					{
						errorHandler.Report(errorHandler.ERROR, errorHandler.IMAGE_EXISTS);
						errorHandler.PrintReport();
						throw new Error('Images exists with read only property.');
					}
				}
				var success  = bitmapItem.exportToFile( imageURL );
				if ( success != true )
				{
					errorHandler.Report( errorHandler.ERROR, errorHandler.CAN_NOT_EXPORT_BITMAP );
					return null;
				}
				gExportedImagesMap[ imageFileName ] = true;
			}			
			
			if(gEmbedImages=="true" || (RESOURCES_URI == "" && !bitmapItem.sourceFileExists))
			{
				var base64Str = FLfile.readBinary(imageURL);
				
				switch ( bitmapItem.originalCompressionType )
				{
					case IMAGE_COMPRESSION_LOSSLESS:
					{
						this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "data:image/png;base64," + base64Str );
						break;
					}
					case IMAGE_COMPRESSION_PHOTO:
					default:
					{
						this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "data:image/jpg;base64," + base64Str );
					}
				}
				FLfile.remove(imageURL);
			}
			else if(RESOURCES_URI == "")
			{
				if(bitmapItem.sourceFileExists)
				{
					gExportedImagesMap[ imageFileName ] = true;
					var os = fl.version.substr(0,3);
					if(os == "MAC")
					{
						var sourceFilePath = bitmapItem.sourceFilePath;
						var modifiedPath = sourceFilePath.substr(0,8) + "Volumes/" + sourceFilePath.substring(8, sourceFilePath.length);
						this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, modifiedPath );
					}
					else
					{
						this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, bitmapItem.sourceFilePath );
					}
				}
			}
			else
			{
				this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "./" + RESOURCES_FOLDER + "/" + imageFileName );
			}
		}
		
		this.xmlComposer.EndNode();
		
		return this;

	}
	
	
	BitmapExporter.prototype.DoesHaveSupportedImageExtension = function ( imageFileName )
	{
		return (/\.(jpg|jpeg|png)$/i).test( imageFileName );
	}
	
	BitmapExporter.prototype.GetNameWithExtnRemoved =  function ( fileName )
	{
			return fileName.replace ( /\.(.+)$/g, "");
	}
	
	/*****************************************************************/
	
	function TextExporter( xmlComposer )
	{
		this.xmlComposer = xmlComposer;
	}
	
	
	TextExporter.prototype.HandleSpecialChars = function ( string )
	{
		return string
		.replace(/&/g, "&amp;")
		.replace(/</g, "&lt;")
		.replace(/>/g, "&gt;")
		.replace(/"/g, "&quot;");
	}
	
	
	TextExporter.prototype.Export = function ( text )
	{
		var dimensions = text.bounds;
		
		var prevInstanceInfo = gInstanceInfoArray[gInstanceInfoArray.length-1 ];
		var colorTransform = prevInstanceInfo.colorTransform;

        if ( gTextFullSupport && text.border && text.textType != TEXT_TYPE_STATIC ) 
		{
			var fillColor = "#FFFFFF";
            var strokeColor = "#000000"
            var fillOpacity = 1.0;
            var strokeOpacity = 1.0;

            if ( colorTransform.HasColorTransform () )
            {
                fillColor = colorTransform.TransformColor ( fillColor );
                if ( fillColor.length > 7 )  //if without alpha, we should have # and RGB each with 2 chars
			    {
					    //the alpha value starts at index 7. The string is a Hex string
				    fillOpacity = parseInt ( fillColor.slice ( 7 ), 16 ) /  255.0; 
				    fillColor = fillColor.slice ( 0, 7);
			    }

                strokeColor = colorTransform.TransformColor ( strokeColor );
                if ( strokeColor.length > 7 )  //if without alpha, we should have # and RGB each with 2 chars
			    {
					    //the alpha value starts at index 7. The string is a Hex string
				    strokeOpacity = parseInt ( fillColor.slice ( 7 ), 16 ) /  255.0; 
				    strokeColor = fillColor.slice ( 0, 7);
			    }
            }

            this.xmlComposer.StartNode ( SVG_RECT );
			this.xmlComposer.AddAttribute ( SVG_COORD_X, dimensions.x);
			this.xmlComposer.AddAttribute ( SVG_COORD_Y,  dimensions.y);
			this.xmlComposer.AddAttribute ( SVG_WIDTH, dimensions.width );
			this.xmlComposer.AddAttribute ( SVG_HEIGHT, dimensions.height );
			this.xmlComposer.AddAttribute ( SVG_FILL_ATTRIB, fillColor );
            if ( fillOpacity != 1.0 ) 
            {
                this.xmlComposer.AddAttribute ( SVG_FILL_OPACITY_ATTRIB, fillOpacity );  
            }

			this.xmlComposer.AddAttribute ( SVG_STROKE_ATTRIB, strokeColor );
            if ( strokeOpacity != 1.0 ) 
            {
                this.xmlComposer.AddAttribute ( SVG_STROKE_OPACITY_ATTRIB, strokeOpacity );  
            }
			this.xmlComposer.EndNode();	
		}
		
		var mask = new MaskExporter ( this.xmlComposer );
		mask.BeginMask(true);

		this.xmlComposer.StartNode ( SVG_RECT );
		this.xmlComposer.AddAttribute ( SVG_COORD_X, dimensions.x );
		this.xmlComposer.AddAttribute ( SVG_COORD_Y,  dimensions.y);
		this.xmlComposer.AddAttribute ( SVG_WIDTH, dimensions.width );
		this.xmlComposer.AddAttribute ( SVG_HEIGHT, dimensions.height );
		this.xmlComposer.AddAttribute ( SVG_FILL_ATTRIB, "#FFFFFF" );
		this.xmlComposer.AddAttribute ( SVG_STROKE_ATTRIB, SVG_STROKE_NONE );
		this.xmlComposer.EndNode();	
		
		mask.EndMask();
	
		this.xmlComposer.StartNode ( SVG_TEXT );
		
		this.xmlComposer.AddAttribute ( SVG_CLIP_PATH_ATTRIBUTE, SVG_URL +"(#"+ mask.formattedName +")" );
		
        var vertical = false;
        if ( text.orientation == TEXT_ORIENTATION_HORIZONTAL )
		{
			this.xmlComposer.AddAttribute ( SVG_TEXT_WRITING_MODE, SVG_TEXT_WRITING_MODE_LR );
		}
		else
		{
			this.xmlComposer.AddAttribute ( SVG_TEXT_WRITING_MODE, SVG_TEXT_WRITING_MODE_TB );
            vertical = true;
		}

		var nextTextRunIndex = 0;
		var lengthRemainingInCurrentRun = 0;
		var currentTextRun;
		var nextExportIndexInCurrentRun = 0;
		
		var singleLine = false;
		var password = false;

		if ( text.textType != TEXT_TYPE_STATIC )
		{
			singleLine = (text.lineType == TEXT_LINE_TYPE_SINGLE  ? true : false);
			password = (text.lineType == TEXT_LINE_TYPE_PASSWORD ? true : false);
			if ( password )
			{
				singleLine = true;
			}
			if(gTextFullSupport && text.renderAsHTML)
			{
				if(!unsupportedFeature)
					unsupportedFeature = true;
			}
		}
		if( gTextFullSupport && text.fontRenderingMode == "bitmap")
		{
			if(!unsupportedFeature)
				unsupportedFeature = true;
		}
		var startingNewLine = true;
		
		for ( var i = 0;  i <  text.textLines.length; i++ )
		{
			var textLine =  text.textLines[i];
			var lengthToRender = textLine.length;
			if ( !singleLine )
			{
				startingNewLine = true;
			}

			while ( lengthToRender )
			{
				var exportLength = 0;
				if ( !lengthRemainingInCurrentRun )
				{
					currentTextRun = text.textRuns[nextTextRunIndex];
					//The last characters such as new line may not be represented in textRun
					if ( !currentTextRun )
					{	
						break;
					}
					lengthRemainingInCurrentRun = currentTextRun.length;
					nextExportIndexInCurrentRun = 0;
					nextTextRunIndex++;
				}
				
				if ( lengthRemainingInCurrentRun > lengthToRender )
				{
					exportLength = lengthToRender;
					lengthRemainingInCurrentRun -= 	lengthToRender;
					lengthToRender = 0;
				}
				else
				{
					exportLength = lengthRemainingInCurrentRun;
					lengthToRender -= lengthRemainingInCurrentRun;
					lengthRemainingInCurrentRun = 0;
				}

				var characters = currentTextRun.characters.slice ( nextExportIndexInCurrentRun, nextExportIndexInCurrentRun+exportLength);
				nextExportIndexInCurrentRun += exportLength;
				
				var textAttrs = currentTextRun.textAttrs;
				
				if(textAttrs.alignment == "justify")
				{
					if(!unsupportedFeature)
						unsupportedFeature = true;
				}
				
	
				this.xmlComposer.StartNode(SVG_TSPAN, true);   //in text we dont introduce any text
				
				if ( startingNewLine )
				{
					this.xmlComposer.AddAttribute ( SVG_COORD_X, textLine.baseline.x);
					this.xmlComposer.AddAttribute ( SVG_COORD_Y, textLine.baseline.y);
					startingNewLine = false;
				}
				
				var baselineShift = 0;
				var superSubScale = 1;
				
				switch ( textAttrs.characterPosition )
				{
					case CHAR_POSITION_SUBSCRIPT: baselineShift = -35; superSubScale = 0.65; break;
					case CHAR_POSITION_SUPERSCRIPT: baselineShift = 35; superSubScale = 0.65; break;
					default:  baselineShift = 0;
				}
				this.xmlComposer.AddAttribute ( SVG_BASELINE_SHIFT, baselineShift+"%");

                if ( text.orientation != TEXT_ORIENTATION_HORIZONTAL )
				{
					if ( textAttrs.rotation )
					{
						this.xmlComposer.AddAttribute ( SVG_TEXT_GLYPH_ORIENTATION_VERTICAL, 90 );
					}
					else
					{
						this.xmlComposer.AddAttribute ( SVG_TEXT_GLYPH_ORIENTATION_VERTICAL, 0 );
					}
				}
				
			    if ( gTextFullSupport && !text.useDeviceFonts )
                {
                   var fontName =  textAttrs.face;
                   fontName = FormatName(fontName);

                   if ( textAttrs.bold ) 
                   {
                        fontName += "_" + FONTNAME_BOLD; 
                   }
                   if ( textAttrs.italic )
                   {
                         fontName += "_" + FONTNAME_ITALIC;
                   }
                   if ( !textAttrs.bold && !textAttrs.italic )
                   {
                        fontName += "_" + FONTNAME_REGULAR;
                   }

                   fontName = fontName.replace( / /g, "");
				   
				   if ( textAttrs.autoKern )
				   {
						this.xmlComposer.AddAttribute ( SVG_KERNING, SVG_KERNING_AUTO );	
				   }
                   else
                   {
                        this.xmlComposer.AddAttribute ( SVG_KERNING, 0 );	
                   }

                   var fontTable = null;
                   if ( gFontTableDefinitions.IsItemPresent ( fontName ) )
                   {
                        fontTable = gFontTableDefinitions.GetResourceItem ( fontName );
                   }
                   else
                   {
                        fontTable = new FontTableExporter( fontName, textAttrs.face, textAttrs.bold,  textAttrs.italic);
                        gFontTableDefinitions.AddResourceItem ( fontTable );
                   }
                   
				   if ( password )
				   {
					   fontTable.AddCharactersForExport ( characters.replace(/./g, "*"), vertical);
				   }
				   else
				   {	
				   		fontTable.AddCharactersForExport ( characters, vertical );
				   }
                   this.xmlComposer.AddAttribute ( SVG_FONT_FAMILY, fontTable.formattedName );
                    if ( textAttrs.italic )
				    {
					    this.xmlComposer.AddAttribute ( SVG_FONT_STYLE, SVG_ITALIC );
				    }
				
				    if ( textAttrs.bold )
				    {
					    this.xmlComposer.AddAttribute ( SVG_FONT_WEIGHT, SVG_BOLD );
				    }
                } // !text.useDeviceFonts
                else
                {

                    this.xmlComposer.AddAttribute ( SVG_FONT_FAMILY, textAttrs.face );
                    if ( textAttrs.italic )
				    {
					    this.xmlComposer.AddAttribute ( SVG_FONT_STYLE, SVG_ITALIC );
				    }
				
				    if ( textAttrs.bold )
				    {
					    this.xmlComposer.AddAttribute ( SVG_FONT_WEIGHT, SVG_BOLD );
				    }
                   
                }
                
                this.xmlComposer.AddAttribute ( SVG_FONT_SIZE, textAttrs.size * superSubScale);
				
				var fillColor = textAttrs.fillColor;

                if ( colorTransform.HasColorTransform() )
		        {
                    fillColor = colorTransform.TransformColor ( fillColor );
		        }

				var opacity =  1.0;
			
				if ( fillColor.length > 7 )  //if without alpha, we should have # and RGB each with 2 chars
				{
					 //the alpha value starts at index 7. The string is a Hex string
					opacity = parseInt ( fillColor.slice ( 7 ), 16 ) /  255.0; 
					fillColor = fillColor.slice ( 0, 7);
				}
	
				this.xmlComposer.AddAttribute ( SVG_FILL_ATTRIB, fillColor );
				if ( opacity !=  1.0 )
				{
					this.xmlComposer.AddAttribute ( SVG_FILL_OPACITY_ATTRIB,  opacity );
				}

				if ( gTextFullSupport && textAttrs.letterSpacing )
				{
					this.xmlComposer.AddAttribute ( SVG_LETTER_SPACING, textAttrs.letterSpacing );
				}
				

                this.xmlComposer.AddAttribute ( SVG_XML_SPACE, SVG_XML_SPACE_PRESERVE );

                characters = characters.replace(/(\r\n|\n|\r)/gm,"");

                if ( password )
				{
					this.xmlComposer.WriteContent (characters.replace(/./g, "*"));
				}
				else
				{
					this.xmlComposer.WriteContent (this.HandleSpecialChars( characters ));
				}

			   this.xmlComposer.EndNode(true);  //for each tspan

			}
		}

		this.xmlComposer.EndNode();	
	}

	/*****************************************************************/
    function FontTableExporter( name, faceName, isBold,  isItalic )
	{
		ResourceItem.call( this ); //call the parent
        this.originalName = name;
        this.formattedName =  this.originalName;
        this.characters = "";

        this.faceName =  faceName;
        this.isBold = isBold;
        this.isItalic =  isItalic;
        this.vertical = false;
	}
	
	FontTableExporter.prototype = new ResourceItem();
	FontTableExporter.prototype.constructor = FontTableExporter;

    FontTableExporter.prototype.AddCharactersForExport = function( characters, vertical )
    {
         this.characters += characters.replace(/(\r\n|\n|\r)/gm,"");
         if ( vertical ) this.vertical = true;
    }
	
    FontTableExporter.prototype.Export = function()
    {
        var fontTable = fl.createFontTable ( this.characters, this.faceName, this.isBold, this.isItalic, this.vertical );
        
        this.xmlComposer.StartNode ( SVG_FONT );
 	 	this.xmlComposer.AddAttribute ( SVG_FONT_HORIZ_ADVANCE_X, 2048 );
        
        if ( this.vertical  )
        {
            this.xmlComposer.AddAttribute ( SVG_VERT_ORIGIN_X, 0 );
            this.xmlComposer.AddAttribute ( SVG_VERT_ORIGIN_Y, 0 );
        }

        //Comment
        this.xmlComposer.WriteComment ( fontTable.fontLicensingInfo.copyrightInfo + "\nURL for more Information: " + fontTable.fontLicensingInfo.URL);
		

        //About the font face
        this.xmlComposer.StartNode ( SVG_FONT_FACE );
        this.xmlComposer.AddAttribute ( SVG_FONT_FAMILY, this.originalName );
        this.xmlComposer.AddAttribute ( SVG_FONT_UNITS_PER_EM, fontTable.EMSquare );
        this.xmlComposer.AddAttribute ( SVG_ASCENT, fontTable.ascent );
        this.xmlComposer.AddAttribute ( SVG_DESCENT, fontTable.descent );
       
        this.xmlComposer.EndNode ();
		
		var kernTable = fontTable.hKernMap;
		if ( kernTable )
		{
			//We want to embed the kerning table.
			for(var i=0; i<kernTable.length; i++)
			{
				this.xmlComposer.StartNode ( SVG_HKERN );
				var first = IsSpecialChar( kernTable[i].first ) ? "&#"+kernTable[i].first+";"  : String.fromCharCode ( kernTable[i].first );
				var second = IsSpecialChar( kernTable[i].second ) ? "&#"+kernTable[i].second+";"  : String.fromCharCode ( kernTable[i].second );
				this.xmlComposer.AddAttribute ( SVG_KERN_U1, first);
				this.xmlComposer.AddAttribute ( SVG_KERN_U2, second);
				this.xmlComposer.AddAttribute ( SVG_KERN_K, -kernTable[i].kernAmount);
				this.xmlComposer.EndNode();	
			}
			
		}

   	    kernTable = fontTable.vKernMap;
		if ( kernTable )
		{
			//We want to embed the kerning table.
			for(var i=0; i<kernTable.length; i++)
			{
				this.xmlComposer.StartNode ( SVG_VKERN );
				var first = IsSpecialChar( kernTable[i].first ) ? "&#"+kernTable[i].first+";"  : String.fromCharCode ( kernTable[i].first );
				var second = IsSpecialChar( kernTable[i].second ) ? "&#"+kernTable[i].second+";"  : String.fromCharCode ( kernTable[i].second );
				this.xmlComposer.AddAttribute ( SVG_KERN_U1, first);
				this.xmlComposer.AddAttribute ( SVG_KERN_U2, second);
				this.xmlComposer.AddAttribute ( SVG_KERN_K, -kernTable[i].kernAmount);
				this.xmlComposer.EndNode();	
			}
			
		}


        var glyphMap = fontTable.glyphMap;

        for ( var i = 0; i < glyphMap.length;  i++ )
        {
            var glyphData = glyphMap[i];

            //Write the glyph
            this.xmlComposer.StartNode ( SVG_GLYPH );
            
			if ( IsSpecialChar( glyphData.charCode ) )
			{
				this.xmlComposer.AddAttribute (SVG_GLYPH_UNICODE,  "&#"+glyphData.charCode+";" );
			}
			else
			{
				this.xmlComposer.AddAttribute (SVG_GLYPH_UNICODE,  String.fromCharCode(glyphData.charCode) );
			}
			
            this.xmlComposer.AddAttribute ( SVG_FONT_HORIZ_ADVANCE_X, glyphData.hAdvance );
 
           var glyph = glyphData.glyph;

            var strokesExporter =  new StrokesContorExporter ( this.xmlComposer, true );
            strokesExporter.Export ( glyph, 0, null );

            this.xmlComposer.EndNode();
        }
        this.xmlComposer.EndNode (); //for svg font

    }

	
	/*****************************************************************/
	
	function PatternDefintion()
	{
		ResourceItem.call( this ); //call the parent	
	}
	
	PatternDefintion.prototype = new ResourceItem();
	PatternDefintion.prototype.constructor = PatternDefintion;
	
	PatternDefintion.prototype.Export = function ( fill )
	{
		//TODO: What if there was item which was already selected?
		var library = gDoc.library;
		library.selectItem( fill.bitmapPath );
		var bitmapItem = library.getSelectedItems()[0];
		library.selectNone();

		var transformMatrix = fill.matrix;
		
		this.xmlComposer.StartNode ( SVG_PATTERN );
		
		this.originalName = PrepareFillName ( fill );
		gPatternCounter++;
		this.formattedName = FormatName(PATTERN + "_" + gPatternCounter);
		gNamesUsed.AddName(this.formattedName);
		this.xmlComposer.AddAttribute ( SVG_ID, this.formattedName );
		this.xmlComposer.AddAttribute ( SVG_PATTERN_UNIT, SVG_USER_SPACEON_USE );
		
		var matrix = {};
		matrix.a = transformMatrix.a / 20;
		matrix.b = transformMatrix.b / 20;
		matrix.c = transformMatrix.c / 20;
		matrix.d = transformMatrix.d / 20;
		matrix.tx = transformMatrix.tx;
		matrix.ty = transformMatrix.ty;
		
		this.xmlComposer.AddAttribute ( SVG_PATTERN_TRANSFORM, GetMatrixString( matrix ) );
		this.xmlComposer.AddAttribute ( SVG_WIDTH, bitmapItem.hPixels );
		this.xmlComposer.AddAttribute ( SVG_HEIGHT, bitmapItem.vPixels );
		
		var bitmapExporter = new BitmapExporter ();
		var afterExport = bitmapExporter.Export ( bitmapItem );
		
		if ( afterExport )
		{
			this.xmlComposer.StartNode( SVG_USE );
			this.xmlComposer.AddAttribute ( SVG_XLINK_HREF, "#"+ afterExport.formattedName );
			this.xmlComposer.EndNode();
		}	
		
		this.xmlComposer.EndNode();
	}
	
	/*****************************************************************/
	
	
	function GradientDefinition( )
	{
		ResourceItem.call( this ); //call the parent
	}
	
	GradientDefinition.prototype = new ResourceItem();
	GradientDefinition.prototype.constructor = GradientDefinition;
	
	GradientDefinition.prototype.Export = function ( fill )
	{
		var colorArray = fill.colorArray;
		var posArray = fill.posArray;
		
		if ( colorArray.length != posArray.length )
		{
			errorHandler.Report ( errorHandler.ERROR, errorHandler.MALFORMED_FILL);
			return;
		}
		
		var fillStyle = SVG_LINEAR_GRADIENT;
		if ( fill.style == RADGRADIENTFILL ) fillStyle = SVG_RADIAL_GRADIENT;
		
		this.xmlComposer.StartNode ( fillStyle );
		
		var myName  = PrepareFillName( fill );
		this.originalName = myName;
		gGradientCounter++;
		this.formattedName = FormatName(GRADIENT + "_" + gGradientCounter);
		gNamesUsed.AddName(this.formattedName);
		this.xmlComposer.AddAttribute ( SVG_ID,   this.formattedName);
		
		var matrix =  fill.matrix;
		this.xmlComposer.AddAttribute ( SVG_GRADIENT_UNITS, SVG_USER_SPACEON_USE);
		
	
		if (  fillStyle == SVG_LINEAR_GRADIENT )
		{
			
			var pt1 = TransformPoint(matrix, { x: -GRADIENT_VECTOR_CONSTANT/20, y: 0} );
			var pt2 = TransformPoint(matrix ,{ x: GRADIENT_VECTOR_CONSTANT/20, y: 0} );
			
			this.xmlComposer.AddAttribute ( SVG_GRADIENT_VECTOR_X1, pt1.x );
			this.xmlComposer.AddAttribute ( SVG_GRADIENT_VECTOR_Y1, pt1.y );
			this.xmlComposer.AddAttribute ( SVG_GRADIENT_VECTOR_X2, pt2.x );
			this.xmlComposer.AddAttribute ( SVG_GRADIENT_VECTOR_Y2, pt2.y );
		}
		else
		{
			//Radial Gradient	
			var pt1 = TransformPoint(matrix, { x:0, y:0 } );  //centre point of circle
			var pt2 = TransformPoint(matrix, { x:GRADIENT_VECTOR_CONSTANT/20, y:0} );   /* p2 us peripheral point */
			
			var xd = pt1.x - pt2.x;
			var yd = pt1.y - pt2.y;
			var r = Math.sqrt( xd * xd + yd * yd );
			
			var a = Math.atan2( yd, xd );
			var focusPointRatio = fill.focalPoint / 255.0;
			var fx = -r * focusPointRatio * Math.cos( a );
			var fy = -r * focusPointRatio * Math.sin( a );
			
			this.xmlComposer.AddAttribute ( RADIAL_GRADIENT_CX, 0 );
			this.xmlComposer.AddAttribute ( RADIAL_GRADIENT_CY, 0 );
			this.xmlComposer.AddAttribute ( RADIAL_GRADIENT_R, r );
			
			this.xmlComposer.AddAttribute ( RADIAL_GRADIENT_FX, fx );
			this.xmlComposer.AddAttribute ( RADIAL_GRADIENT_FY, fy );
			
			var scaleFactor =  (GRADIENT_VECTOR_CONSTANT/20) / r;
			var matrix1 = {};
			matrix1.a = matrix.a * scaleFactor;
			matrix1.b = matrix.b * scaleFactor;
			matrix1.c = matrix.c * scaleFactor;
			matrix1.d = matrix.d * scaleFactor;
			matrix1.tx =  matrix.tx;
			matrix1.ty = matrix.ty;
			
			this.xmlComposer.AddAttribute ( SVG_GRADIENT_TRANSFORM, GetMatrixString(matrix1) );
		}
		
		//specify the spread method
		var spread = SVG_SPREAD_PAD;

		switch ( fill.overflow )
		{
			case GRADIENT_EXTEND:  spread = SVG_SPREAD_PAD; break;
			case GRADIENT_REPEAT:  spread = SVG_SPREAD_REPEAT; break;
			case GRADIENT_REFLECT: spread = SVG_SPREAD_REFLECT;
		}
	
		this.xmlComposer.AddAttribute (  SVG_SPREAD_METHOD,  spread);
		
		for ( var num = 0; num < colorArray.length;  num++ )
		{
			var percentage = posArray[num] * 100.0 / 255.0;
			
			this.xmlComposer.StartNode ( SVG_GRADIENT_STOP );
			this.xmlComposer.AddAttribute ( SVG_GRADIENT_OFFSET, percentage+"%" );
			
			var fillColor = colorArray[num];
			var opacity =  1.0;
			
			
			if ( fillColor.length > 7 )  //if without alpha, we should have # and RGB each with 2 chars
			{
				 //the alpha value starts at index 7. The string is a Hex string
				opacity = parseInt ( fillColor.slice ( 7 ), 16 ) /  255.0; 
				fillColor = fillColor.slice ( 0, 7);
			}

			this.xmlComposer.AddAttribute ( SVG_GRADIENT_STOP_COLOR,  fillColor );
			if ( opacity !=  1.0 )
			{
				this.xmlComposer.AddAttribute ( SVG_GRADIENT_STOP_OPACITY,  opacity );
			}
			
			this.xmlComposer.EndNode();
		}
		
		this.xmlComposer.EndNode(); //for gradient fill style
	}
	
	/*****************************************************************/
	
	function GetMatrixString( matrix )
	{
		return 	SVG_MATRIX + "( "+ matrix.a + ", "+
												matrix.b + ", "+
												matrix.c + ", "+
												matrix.d + ", "+
												matrix.tx + ","+
												matrix.ty + ") ";
	}
	
	
	function TransformPoint ( matrix, pt )
	{
		var x = pt.x;
		var y = pt.y;
		pt.x = x * matrix.a + y * matrix.c + matrix.tx;
		pt.y = x * matrix.b + y * matrix.d + matrix.ty;
		return pt;
	}
	
	function FormatName (  str  )
	{
		var name = str;
		if ( gMaskMode )
		{
			name = MASK + "_" + str;
		}
		name = name.replace( /\W/g, "_");
		if (/[\d]/.test(name.charAt(0)))
			name = "FL_" + name;
		while(gNamesUsed.IsNameUsed(name) == true)		
		{
			name = name + "_FL";
		}
		return name;
	}
	
	
	/*****************************************************************/
	
	function PrepareFillName ( fill )
	{
		var fillName = fill.style + "_";
		
		if ( gMaskMode )
		{
			return ( fillName + "#FFFFFF" );	
		}
		
		switch ( fill.style )
		{
			case SOLIDFILL:
			{
				fillName += fill.color
				break;
			}
			case LINGRADIENTFILL:
			case RADGRADIENTFILL:
			{
				for ( var num = 0; num < fill.colorArray.length;  num++ )
				{
					fillName += fill.colorArray[num].slice ( 1 );
				}
				break;			
			}
			case BITMAPFILL:
			{
				var bitmapName = fill.bitmapPath.replace( /\W/g, "_");

				fillName += bitmapName + "_";
				break;
			}
		}
		
		if ( fill.matrix )
		{
			fillName += fill.matrix.tx + "_"+fill.matrix.ty + "_" + fill.matrix.a + "_" + fill.matrix.b + "_" + fill.matrix.c + "_" + fill.matrix.d ;
		}

		return fillName;
	}

	/*****************************************************************/
	
	function ArePointsEqual ( point1, point2 )
	{
		if ( point1 == "" || point2 == "" )
		{
			return false;	
		}
		if ( point1.x == point2.x  && point1.y == point2.y )
		{
			return true;				
		}
		return false;
	}
	
	function Point()
	{
		this.x = 0;
		this.y = 0;
	}
	
	function  GetBoundingBox ( x, y, width, height, matrix )
	{
		var points = [];
		var bound;
		
		var point = new Point();
		point.x = x;
		point.y = y;
		points[0] = point;

		point = new Point();
		point.x = x + width;
		point.y = y;
		points[1] = point;
			
		point = new Point();
		point.x = x ;
		point.y = y + height;
		points[2] = point;

		point = new Point();
		point.x = x + width;
		point.y = y + height;
		points[3] = point;
		
		var minX = 16384;
		var minY = 16384;
		var maxX = -16384;
		var maxY = -16384;
		
		for ( var i = 0; i < 4;  i++ )
		{
			var point = TransformPoint ( matrix, points[i] );
			if ( point.x < minX )
				minX = point.x;
			if ( point.x > maxX )
				maxX = point.x;
			if ( point.y < minY )
				minY = point.y;
			if ( point.y > maxY )
				maxY = point.y;
		}
		
		var bound = {};
		bound.x = minX;
		bound.y = minY;
		bound.width = ( maxX - minX );
		bound.height = ( maxY - minY );		
		
		return bound;
	}
	
	/*****************************************************************/
	function CreateIdentityMatrix()
	{
		var matrix = {};
		matrix.a = 1;
		matrix.b = 0;
		matrix.c = 0;
		matrix.d = 1;
		matrix.tx = 0;
		matrix.ty = 0;
		
		return matrix;
	}
	
	function HasScaleFactors( matrix )
	{
		if ( matrix.a != 1 || matrix.b != 0 || matrix.c != 0 && matrix.d != 1 )
			return true;
		else 
			return false;
	}


	function IsIdentityMatrix( matrix )
	{
		if ( matrix.a == 1 && matrix.b == 0 && matrix.c == 0 && matrix.d == 1  && matrix.tx == 0 && matrix.ty == 0)
			return true;
		else 
			return false;
	}

	/*****************************************************************/
	
	function PrintMatrix ( matrix )
	{
		trace ( 	"matrix.a =" +	matrix.a + "matrix.b =" +	matrix.b + 
				 	"matrix.c =" +	matrix.c +  "matrix.d =" +	matrix.d +  
				 	"matrix.tx =" +	matrix.tx +  "matrix.ty =" +	matrix.ty );
		
	}
	/*****************************************************************/
	
	function IsSpecialChar( charCode )
	{
		var str = String.fromCharCode ( charCode );
		if ( str == '&'  || str == '<'  || str == '>' || str == '"' )
		 return true;
		else 
		 return false;		
	}


}());
